<?php

namespace wpforo\classes;

use stdClass;

class Settings extends stdClass {
	public $_SERVER;
	public $info;
	
	public $_general;
	public $_board;
	public $_slugs;
	public $_members;
	public $_profiles;
	public $_rating;
	public $_authorization;
	public $_recaptcha;
	public $_buddypress;
	public $_um;
	public $_forums;
	public $_topics;
	public $_posting;
	public $_components;
	public $_styles;
	public $_tags;
	public $_email;
	public $_subscriptions;
	public $_notifications;
	public $_logging;
	public $_seo;
	public $_antispam;
	public $_akismet;
	public $_rss;
	public $_social;
	public $_legal;
	
	public $general;
	public $board;
	public $slugs = [];
	public $members;
	public $profiles;
	public $rating;
	public $authorization;
	public $recaptcha;
	public $buddypress;
	public $um;
	public $forums;
	public $topics;
	public $posting;
	public $components;
	public $styles;
	public $tags;
	public $email;
	public $subscriptions;
	public $notifications;
	public $logging;
	public $seo;
	public $antispam;
	public $akismet;
	public $rss;
	public $social;
	public $legal;
	
	private function templates_to__slugs( $templates ) {
		foreach( $templates as $template_key => $template ) {
			if( ! wpfkey( $this->_slugs, $template_key ) ) $this->_slugs[ $template_key ] = ( wpfval( $template, 'slug' ) ?: $template_key );
		}
	}
	
	public function __construct() {
		$this->init__SERVER();
		$this->init_defaults();
		$this->init_base();
		add_action( 'wpforo_after_init_base_templates', function( $base_templates ) {
			$this->templates_to__slugs( $base_templates );
			$this->slugs = wpforo_get_option( 'wpforo_base_slugs', array_merge( $this->_slugs, $this->slugs ) );
			$this->slugs = array_map( 'wpforo_fix_url', $this->slugs );
		} );
		add_action( 'wpforo_after_init_member_templates', function( $member_templates ) {
			$this->templates_to__slugs( $member_templates );
			$this->slugs = wpforo_get_option( 'slugs', array_merge( $this->_slugs, $this->slugs ) );
			$this->slugs = array_map( 'wpforo_fix_url', $this->slugs );
		} );
		add_action( 'wpforo_after_init_templates', function( $templates ) {
			$this->templates_to__slugs( $templates );
			$this->slugs = wpforo_get_option( 'slugs', array_merge( $this->_slugs, $this->slugs ) );
			$this->slugs = array_map( 'wpforo_fix_url', $this->slugs );
		} );
		add_action( 'wpforo_after_change_board', function() { $this->init(); } );
	}
	
	private function init__SERVER() {
		$umf           = wpforo_human_size_to_bytes( ini_get( 'upload_max_filesize' ) );
		$pms           = wpforo_human_size_to_bytes( ini_get( 'post_max_size' ) );
		$maxs_min      = min( $umf, $pms );
		$this->_SERVER = [
			'upload_max_filesize'       => $umf,
			'post_max_size'             => $pms,
			'maxs_min'                  => $maxs_min,
			'upload_max_filesize_human' => wpforo_print_size( $umf ),
			'post_max_size_human'       => wpforo_print_size( $pms ),
			'maxs_min_human'            => wpforo_print_size( $maxs_min ),
		];
	}
	
	public function init_info() {
		$this->info = new stdClass();
		$this->init_core();
		$this->init_addons();
	}
	
	private function init_core() {
		$this->info->core = [
			"general"       => [
				"title"                => esc_html__( "General Settings", "wpforo" ),
				"title_original"       => "General Settings",
				"icon"                 => '<svg version="1.1" height="55px" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" viewBox="0 0 122.881 122.88" enable-background="new 0 0 122.881 122.88" xml:space="preserve"><g><path d="M97.977,18.705c-0.33-0.33-0.67-0.568-1.02-0.716c-0.328-0.139-0.711-0.207-1.15-0.207c-0.441,0-0.832,0.07-1.164,0.211 c-0.344,0.145-0.676,0.379-0.994,0.702l-0.021,0.021l-6.207,6.188c-0.93,0.925-2.367,1.028-3.41,0.313 c-0.73-0.472-1.508-0.927-2.324-1.368c-0.865-0.468-1.713-0.888-2.541-1.268c-0.863-0.394-1.744-0.765-2.641-1.113 c-0.813-0.316-1.719-0.638-2.707-0.964c-1.131-0.373-1.85-1.425-1.85-2.556l0,0V8.43c0-0.456-0.072-0.859-0.215-1.21 c-0.143-0.347-0.361-0.67-0.662-0.97c-0.293-0.293-0.611-0.51-0.959-0.651c-0.35-0.143-0.754-0.214-1.211-0.214H56.956 c-0.423,0-0.798,0.068-1.126,0.204c-0.348,0.143-0.689,0.377-1.026,0.703L54.8,6.295c-0.296,0.296-0.514,0.611-0.654,0.946 C54.002,7.584,53.93,7.98,53.93,8.43v8.714c0,1.325-0.957,2.426-2.217,2.651c-0.957,0.228-1.841,0.464-2.646,0.711 c-0.882,0.27-1.769,0.577-2.658,0.924c-0.044,0.017-0.088,0.033-0.132,0.047c-0.777,0.302-1.596,0.652-2.456,1.049 c-0.874,0.403-1.688,0.812-2.442,1.223c-1.063,0.579-2.345,0.364-3.165-0.443l-0.001,0.001l-6.874-6.775 c-0.021-0.021-0.042-0.042-0.063-0.063l0,0.001c-0.293-0.31-0.601-0.537-0.924-0.678c-0.31-0.135-0.677-0.204-1.105-0.204 c-0.425,0-0.805,0.072-1.139,0.215c-0.366,0.156-0.716,0.4-1.052,0.729l-0.002,0.002l-8.339,8.358l-0.005,0.006l0.005,0.005 c-0.342,0.342-0.584,0.682-0.727,1.02c-0.139,0.328-0.208,0.711-0.208,1.149c0,0.442,0.07,0.832,0.211,1.166 c0.145,0.343,0.378,0.675,0.702,0.993l0.021,0.021l6.188,6.208c0.925,0.93,1.028,2.367,0.313,3.41 c-0.472,0.731-0.927,1.508-1.367,2.324c-0.468,0.866-0.888,1.712-1.268,2.542c-0.395,0.862-0.766,1.744-1.115,2.641 c-0.316,0.812-0.637,1.718-0.963,2.706c-0.373,1.132-1.426,1.85-2.556,1.85v0.001H8.431c-0.458,0-0.86,0.071-1.211,0.213 c-0.346,0.142-0.666,0.358-0.958,0.651L6.251,51.81l0,0c-0.293,0.293-0.51,0.612-0.652,0.958c-0.143,0.351-0.213,0.754-0.213,1.211 v11.946c0,0.422,0.067,0.798,0.203,1.125c0.144,0.348,0.378,0.689,0.704,1.027l0.003,0.002c0.295,0.296,0.611,0.515,0.945,0.654 c0.344,0.144,0.74,0.216,1.19,0.216h8.713c1.325,0,2.426,0.957,2.651,2.218c0.228,0.956,0.464,1.841,0.711,2.646 c0.271,0.882,0.577,1.768,0.924,2.657c0.326,0.847,0.696,1.727,1.108,2.639c0.421,0.933,0.833,1.783,1.233,2.552 c0.557,1.066,0.326,2.336-0.486,3.143l0.002,0.002l-6.776,6.756l-0.043,0.042h0.001c-0.31,0.294-0.537,0.602-0.677,0.924 c-0.136,0.31-0.204,0.678-0.204,1.106c0,0.425,0.072,0.804,0.214,1.138c0.156,0.366,0.4,0.717,0.73,1.052l8.309,8.405 c0.325,0.301,0.664,0.521,1.019,0.66c0.356,0.141,0.76,0.21,1.212,0.21c0.456,0,0.867-0.071,1.229-0.214 c0.348-0.137,0.675-0.347,0.978-0.631l6.159-6.257c0.929-0.944,2.384-1.057,3.438-0.329c0.729,0.47,1.504,0.923,2.318,1.362 c0.866,0.469,1.712,0.889,2.542,1.268c0.862,0.396,1.744,0.767,2.641,1.115c0.812,0.315,1.717,0.637,2.706,0.963 c1.132,0.373,1.85,1.426,1.85,2.556h0.001v9.518c0,0.456,0.071,0.859,0.214,1.21c0.142,0.347,0.358,0.666,0.651,0.959l0.005,0.006 l0.005-0.006c0.587,0.587,1.307,0.876,2.169,0.876h11.944c0.424,0,0.799-0.067,1.127-0.203c0.348-0.144,0.689-0.378,1.025-0.704 l0.004-0.003c0.295-0.296,0.514-0.611,0.654-0.945c0.145-0.344,0.217-0.739,0.217-1.189v-8.715c0-1.324,0.955-2.426,2.217-2.65 c0.955-0.228,1.84-0.465,2.646-0.711c0.881-0.271,1.768-0.577,2.656-0.923c0.848-0.327,1.729-0.697,2.643-1.109 c0.93-0.421,1.781-0.832,2.547-1.232c1.068-0.557,2.338-0.325,3.145,0.486l0.002-0.002l6.756,6.775l0.041,0.043v-0.001 c0.297,0.312,0.605,0.538,0.926,0.678c0.314,0.135,0.693,0.204,1.145,0.204c0.443,0,0.828-0.072,1.16-0.215 c0.334-0.145,0.645-0.368,0.928-0.667c0.031-0.034,0.064-0.067,0.1-0.1l8.369-8.272c0.299-0.324,0.52-0.664,0.658-1.019 c0.141-0.356,0.211-0.761,0.211-1.213c0-0.456-0.07-0.867-0.213-1.229c-0.139-0.348-0.348-0.674-0.633-0.977l-6.256-6.16 c-0.945-0.929-1.057-2.383-0.33-3.438c0.471-0.729,0.924-1.504,1.363-2.317c0.469-0.865,0.889-1.712,1.268-2.542 c0.395-0.862,0.766-1.744,1.115-2.641c0.314-0.812,0.637-1.718,0.963-2.707c0.373-1.132,1.426-1.85,2.557-1.85v-0.001h9.516 c0.457,0,0.861-0.071,1.211-0.214c0.346-0.142,0.666-0.358,0.959-0.65l0.01-0.011l0,0c0.293-0.294,0.512-0.613,0.652-0.96 c0.143-0.35,0.213-0.753,0.213-1.21V56.955c0-0.422-0.066-0.798-0.203-1.126c-0.143-0.347-0.377-0.689-0.703-1.027l-0.004-0.003 c-0.295-0.296-0.611-0.514-0.945-0.654c-0.342-0.144-0.738-0.216-1.189-0.216h-8.713c-1.342,0-2.453-0.979-2.658-2.262 c-0.211-0.843-0.449-1.698-0.719-2.566c-0.26-0.833-0.564-1.714-0.92-2.645c-0.018-0.045-0.033-0.091-0.047-0.137 c-0.338-0.885-0.684-1.717-1.039-2.496c-0.385-0.843-0.793-1.658-1.225-2.446c-0.578-1.063-0.363-2.345,0.443-3.165v-0.001 l6.775-6.874c0.021-0.021,0.041-0.042,0.063-0.063v0c0.311-0.293,0.537-0.601,0.678-0.923c0.135-0.31,0.205-0.677,0.205-1.105 c0-0.425-0.072-0.805-0.215-1.139c-0.156-0.366-0.4-0.716-0.73-1.052l-0.002-0.002l-8.359-8.339L97.977,18.705L97.977,18.705 L97.977,18.705z M99.039,13.044c1.002,0.423,1.912,1.042,2.732,1.862l0.002,0.001l8.381,8.362l0.021,0.021 c0.807,0.82,1.424,1.723,1.846,2.713c0.436,1.021,0.656,2.102,0.656,3.243c0,1.158-0.225,2.24-0.668,3.251 c-0.432,0.992-1.063,1.874-1.879,2.653l-5.367,5.445c0.158,0.329,0.316,0.662,0.469,0.999c0.43,0.938,0.824,1.878,1.182,2.819 c0.02,0.042,0.037,0.085,0.053,0.129c0.357,0.937,0.699,1.933,1.025,2.981c0.105,0.338,0.207,0.678,0.305,1.021h6.652 c1.156,0,2.238,0.212,3.252,0.636c0.996,0.417,1.896,1.032,2.699,1.841l0.057,0.057c0.791,0.818,1.391,1.72,1.799,2.71 c0.418,1.007,0.625,2.062,0.625,3.167V68.9c0,1.149-0.203,2.225-0.613,3.23c-0.41,1.003-1.021,1.914-1.838,2.733l-0.004,0.003 l-0.006,0.007l-0.004,0.004c-0.82,0.817-1.73,1.43-2.736,1.84c-1.006,0.409-2.082,0.613-3.23,0.613h-7.609 c-0.133,0.358-0.271,0.728-0.42,1.106c-0.381,0.979-0.793,1.958-1.24,2.936c-0.463,1.011-0.934,1.966-1.414,2.857 c-0.158,0.292-0.32,0.583-0.486,0.874l4.75,4.676c0.035,0.033,0.068,0.066,0.104,0.102c0.779,0.824,1.365,1.739,1.76,2.739 c0.396,1.004,0.592,2.068,0.592,3.186c0,1.114-0.195,2.169-0.588,3.169c-0.393,0.997-0.973,1.908-1.744,2.734v-0.001 c-0.027,0.028-0.055,0.058-0.084,0.086l-8.43,8.333c-0.791,0.829-1.682,1.463-2.682,1.893c-1.01,0.436-2.102,0.657-3.283,0.657 c-1.156,0-2.246-0.226-3.27-0.667c-1.014-0.439-1.91-1.074-2.691-1.898l-5.344-5.358c-0.357,0.172-0.707,0.335-1.047,0.489 c-0.91,0.41-1.883,0.815-2.916,1.214c-0.971,0.378-1.984,0.727-3.035,1.049c-0.35,0.107-0.695,0.209-1.035,0.306v6.638 c0,1.155-0.213,2.238-0.637,3.251c-0.418,0.997-1.031,1.896-1.842,2.7l-0.057,0.058c-0.818,0.789-1.721,1.388-2.709,1.798 c-1.008,0.417-2.063,0.624-3.168,0.624H53.979c-2.349,0-4.333-0.81-5.977-2.454l0.005-0.005c-0.82-0.821-1.435-1.734-1.845-2.741 c-0.41-1.006-0.614-2.081-0.614-3.23v-7.609c-0.359-0.133-0.728-0.272-1.107-0.42c-0.978-0.381-1.957-0.793-2.935-1.241 c-1.011-0.462-1.966-0.934-2.858-1.415c-0.291-0.157-0.582-0.319-0.873-0.485l-4.676,4.75c-0.033,0.035-0.067,0.069-0.102,0.103 c-0.824,0.781-1.739,1.367-2.739,1.761c-1.004,0.396-2.068,0.593-3.186,0.593c-1.114,0-2.17-0.195-3.169-0.589 c-0.997-0.393-1.907-0.973-2.734-1.743l0.001-0.002c-0.029-0.026-0.058-0.054-0.085-0.083l-8.381-8.478 c-0.806-0.82-1.423-1.724-1.845-2.714c-0.435-1.022-0.656-2.102-0.656-3.242c0-1.158,0.225-2.24,0.667-3.252 c0.437-0.999,1.071-1.886,1.897-2.67l5.359-5.344c-0.172-0.357-0.336-0.707-0.49-1.05c-0.411-0.909-0.815-1.88-1.213-2.912 c-0.378-0.971-0.728-1.984-1.05-3.036c-0.107-0.351-0.209-0.695-0.306-1.035H8.431c-1.156,0-2.239-0.213-3.251-0.637 c-0.998-0.417-1.897-1.031-2.7-1.841L2.422,71.8c-0.79-0.817-1.389-1.72-1.798-2.709C0.207,68.083,0,67.028,0,65.925V53.979 c0-1.149,0.204-2.225,0.614-3.23c0.41-1.005,1.021-1.916,1.839-2.735l0.001-0.001l0.006-0.005l0.005-0.005 c0.82-0.818,1.73-1.43,2.736-1.84c1.005-0.41,2.081-0.613,3.23-0.613h7.609c0.132-0.359,0.272-0.728,0.42-1.107 c0.38-0.978,0.793-1.957,1.241-2.936c0.462-1.011,0.933-1.966,1.415-2.857c0.16-0.297,0.326-0.594,0.496-0.891l-4.681-4.696 l-0.002-0.001c-0.827-0.813-1.453-1.728-1.88-2.739c-0.437-1.032-0.653-2.118-0.653-3.249c0-1.129,0.215-2.205,0.649-3.232 c0.43-1.017,1.05-1.932,1.863-2.745l0.005,0.005l8.356-8.376l0.021-0.021c0.82-0.806,1.723-1.423,2.713-1.845 c1.022-0.435,2.102-0.657,3.243-0.657c1.157,0,2.239,0.225,3.251,0.667c0.992,0.433,1.874,1.063,2.653,1.88l5.441,5.362 c0.327-0.162,0.653-0.317,0.978-0.467c0.837-0.386,1.761-0.775,2.772-1.167c0.043-0.02,0.087-0.038,0.131-0.055 c0.971-0.378,1.985-0.728,3.037-1.05c0.35-0.107,0.695-0.209,1.035-0.306V8.43c0-1.155,0.213-2.238,0.637-3.25 c0.417-0.998,1.032-1.897,1.841-2.701l0.057-0.057c0.818-0.79,1.72-1.388,2.709-1.797C54.796,0.207,55.851,0,56.956,0H68.9 c1.148,0,2.225,0.203,3.23,0.613c1.01,0.412,1.924,1.027,2.748,1.85c0.814,0.816,1.428,1.728,1.84,2.737 c0.41,1.005,0.613,2.081,0.613,3.229v7.61c0.359,0.133,0.727,0.273,1.107,0.42c0.977,0.38,1.955,0.793,2.934,1.24 c1.012,0.462,1.967,0.934,2.857,1.416c0.299,0.161,0.596,0.326,0.893,0.496l4.695-4.681l0.002-0.001 c0.813-0.827,1.729-1.453,2.738-1.88c1.033-0.437,2.117-0.653,3.248-0.653C96.936,12.396,98.012,12.61,99.039,13.044L99.039,13.044 z M61.44,35.346c1.781,0,3.519,0.17,5.214,0.508c1.666,0.333,3.318,0.844,4.953,1.534l0.018,0.008l0.004-0.008 c1.561,0.674,3.041,1.479,4.441,2.417c1.387,0.929,2.672,1.978,3.85,3.145l0.01,0.011l0,0c1.168,1.179,2.217,2.463,3.146,3.851 c0.938,1.4,1.742,2.88,2.416,4.441c0.031,0.074,0.059,0.148,0.084,0.223c0.646,1.569,1.131,3.152,1.449,4.751 c0.34,1.696,0.508,3.433,0.508,5.213c0,1.781-0.168,3.519-0.508,5.214c-0.332,1.667-0.844,3.317-1.533,4.953l-0.008,0.018 l0.008,0.004c-0.674,1.561-1.479,3.04-2.416,4.44c-0.93,1.388-1.979,2.672-3.146,3.851l-0.01,0.011l0,0 c-1.18,1.168-2.463,2.217-3.852,3.146c-1.398,0.938-2.879,1.742-4.439,2.416c-0.074,0.031-0.148,0.06-0.225,0.084 c-1.568,0.647-3.152,1.131-4.75,1.45c-1.695,0.339-3.434,0.508-5.214,0.508c-1.781,0-3.518-0.169-5.213-0.508 c-1.667-0.333-3.318-0.845-4.954-1.534l-0.018-0.008l-0.003,0.008c-1.561-0.674-3.042-1.479-4.441-2.416 c-1.388-0.929-2.672-1.978-3.85-3.146l-0.01-0.011l0,0c-1.168-1.18-2.218-2.463-3.146-3.852c-0.937-1.399-1.742-2.88-2.415-4.439 c-0.032-0.074-0.06-0.148-0.085-0.224c-0.647-1.569-1.131-3.152-1.45-4.751c-0.338-1.695-0.508-3.433-0.508-5.214 c0-1.781,0.169-3.518,0.508-5.213c0.333-1.667,0.845-3.318,1.535-4.953l0.007-0.018l-0.007-0.003 c0.673-1.561,1.479-3.041,2.415-4.44c0.929-1.389,1.978-2.672,3.146-3.852l0.011-0.01l0,0c1.178-1.167,2.462-2.216,3.849-3.145 c1.4-0.937,2.881-1.743,4.442-2.417c0.074-0.032,0.148-0.06,0.224-0.084c1.569-0.647,3.151-1.13,4.75-1.449 C57.922,35.516,59.659,35.346,61.44,35.346L61.44,35.346z M65.602,41.134c-1.344-0.269-2.73-0.403-4.162-0.403 c-1.431,0-2.817,0.134-4.162,0.403c-1.306,0.261-2.553,0.635-3.741,1.122c-0.052,0.026-0.105,0.052-0.16,0.075 c-1.298,0.561-2.492,1.205-3.58,1.933c-1.096,0.733-2.11,1.563-3.044,2.487c-0.925,0.935-1.755,1.951-2.489,3.048 c-0.728,1.087-1.372,2.28-1.932,3.578l-0.007-0.003c-0.521,1.237-0.917,2.539-1.191,3.905c-0.268,1.344-0.402,2.73-0.402,4.161 s0.134,2.817,0.402,4.162c0.261,1.306,0.636,2.553,1.123,3.741c0.026,0.053,0.052,0.105,0.075,0.16 c0.56,1.298,1.204,2.49,1.931,3.577c0.734,1.097,1.564,2.113,2.49,3.048c0.934,0.925,1.949,1.754,3.045,2.487 c1.087,0.728,2.28,1.372,3.579,1.933l-0.003,0.007c1.238,0.521,2.539,0.918,3.905,1.19c1.344,0.269,2.731,0.402,4.162,0.402 c1.431,0,2.818-0.134,4.162-0.402c1.307-0.261,2.553-0.635,3.742-1.122c0.051-0.026,0.105-0.052,0.16-0.075 c1.297-0.561,2.49-1.205,3.578-1.933c1.096-0.733,2.111-1.563,3.045-2.487c0.926-0.935,1.756-1.95,2.49-3.047 c0.727-1.087,1.371-2.28,1.93-3.578l0.008,0.003c0.521-1.237,0.918-2.539,1.191-3.904c0.268-1.345,0.402-2.731,0.402-4.162 s-0.135-2.817-0.402-4.161c-0.262-1.307-0.635-2.553-1.123-3.742c-0.027-0.052-0.051-0.105-0.076-0.16 c-0.559-1.298-1.203-2.491-1.932-3.578c-0.732-1.097-1.563-2.113-2.488-3.047c-0.934-0.924-1.949-1.754-3.045-2.487 c-1.088-0.728-2.281-1.372-3.578-1.933l0.002-0.007C68.27,41.804,66.969,41.407,65.602,41.134L65.602,41.134z"/></g></svg>',
				"description"          => esc_html__(
					"Here you can manage admin bar displaying options, using WordPress date format in forum and permalinks of general pages. If you have hmore then one forum board, this settings are still are general for all forum boards.",
					"wpforo"
				),
				"description_original" => "Here you can manage admin bar displaying options, using WordPress date format in forum and permalinks of general pages. If you have hmore then one forum board, this settings are still are general for all forum boards.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/general-settings/",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/general.php' );
				},
				"options"              => [
					"admin_bar"      => [
						"type"                 => "checkbox",
						"label"                => esc_html__( "Show Admin Bar for Usergroups", "wpforo" ),
						"label_original"       => "Show Admin Bar for Usergroups",
						"description"          => esc_html__(
							"This option displays the top black bar for accessing to the dashboard. Make sure the regular members can only access to their own profile editing page before enabling this option.",
							"wpforo"
						),
						"description_original" => "This option displays the top black bar for accessing to the dashboard. Make sure the regular members can only access to their own profile editing page before enabling this option.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/general-settings/#show-admin-bar-for-usergroups",
						"variants"             => $this->get_variants_usergroups( 4 ),
					],
					"wp_date_format" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable WordPress Date/Time Format", "wpforo" ),
						"label_original"       => "Enable WordPress Date/Time Format",
						"description"          => esc_html__( "You can manage WordPress date and time format in WordPress Settings > General admin page.", "wpforo" ),
						"description_original" => "You can manage WordPress date and time format in WordPress Settings > General admin page.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/general-settings/#enable-wordpress-date-time-format",
					],
					"fontawesome"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable wpForo Font-Awesome Lib", "wpforo" ),
						"label_original"       => "Enable wpForo Font-Awesome Lib",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'sitewide', 'label' => 'Sitewide' ],
							[ 'value' => 'forum', 'label' => 'Forum' ],
							[ 'value' => 'off', 'label' => 'Off' ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/colors-styles/#enable-wpforo-font-awesome-lib",
					],
					"debug_mode"     => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Debug Mode", "wpforo" ),
						"label_original"       => "Enable Debug Mode",
						"description"          => esc_html__(
							"If you got some issue with wpForo, please enable this option before asking for support, this outputs hidden important information to help us debug your issue.",
							"wpforo"
						),
						"description_original" => "If you got some issue with wpForo, please enable this option before asking for support, this outputs hidden important information to help us debug your issue.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/general-settings/#enable-debug-mode",
					],
				],
			],
			"board"         => [
				"title"                => esc_html__( "Board Settings", "wpforo" ),
				"title_original"       => "Board Settings",
				"icon"                 => '<svg version="1.1" height="55px" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" viewBox="0 0 122.881 95.629" enable-background="new 0 0 122.881 95.629" xml:space="preserve"><g><path d="M112.13,54.992c-0.495-0.348-1.023-0.47-1.595-0.37s-1.034,0.395-1.375,0.889l-1.869,2.649 c-0.675-0.287-1.377-0.518-2.102-0.707s-1.456-0.335-2.188-0.442l-0.605-3.469c-0.102-0.586-0.388-1.043-0.865-1.38 c-0.478-0.336-1.006-0.449-1.591-0.347l-4.354,0.759c-0.557,0.098-1.007,0.382-1.357,0.862c-0.342,0.479-0.464,1.008-0.361,1.593 l0.554,3.177c-0.698,0.298-1.368,0.636-1.999,1.019c-0.64,0.383-1.237,0.796-1.792,1.239l-2.937-2.033 c-0.465-0.346-0.98-0.477-1.559-0.375c-0.571,0.099-1.026,0.399-1.375,0.887l-2.516,3.587c-0.348,0.495-0.469,1.023-0.369,1.594 c0.1,0.572,0.395,1.035,0.888,1.376l2.65,1.869c-0.287,0.675-0.518,1.377-0.708,2.102c-0.189,0.725-0.334,1.456-0.442,2.188 l-3.469,0.605c-0.585,0.102-1.043,0.387-1.379,0.865c-0.336,0.478-0.45,1.006-0.348,1.591l0.76,4.354 c0.097,0.557,0.381,1.007,0.861,1.357c0.479,0.342,1.008,0.464,1.594,0.361l3.176-0.554c0.298,0.698,0.636,1.367,1.019,1.999 c0.384,0.64,0.8,1.251,1.247,1.834l-2.041,2.895c-0.346,0.465-0.476,0.98-0.375,1.559c0.1,0.571,0.399,1.026,0.887,1.375 l3.595,2.558c0.49,0.319,1.016,0.427,1.587,0.327s1.037-0.38,1.383-0.845l1.862-2.693c0.674,0.287,1.376,0.518,2.102,0.707 c0.724,0.19,1.455,0.335,2.188,0.442l0.604,3.47c0.102,0.585,0.388,1.043,0.865,1.379c0.478,0.335,1.006,0.45,1.591,0.348 l4.354-0.76c0.557-0.097,1.007-0.382,1.357-0.861c0.343-0.479,0.464-1.008,0.361-1.594l-0.554-3.177 c0.698-0.298,1.368-0.635,2-1.018c0.639-0.384,1.25-0.8,1.834-1.247l2.894,2.041c0.466,0.346,0.988,0.475,1.573,0.372 c0.593-0.104,1.04-0.401,1.361-0.885l2.558-3.594c0.318-0.49,0.426-1.016,0.327-1.587c-0.101-0.571-0.381-1.037-0.845-1.383 l-2.693-1.861c0.287-0.676,0.518-1.378,0.707-2.103c0.189-0.724,0.334-1.456,0.442-2.188l3.469-0.604 c0.585-0.103,1.043-0.388,1.379-0.866c0.337-0.478,0.45-1.005,0.348-1.591l-0.759-4.354c-0.098-0.557-0.382-1.008-0.862-1.357 c-0.479-0.344-1.008-0.464-1.593-0.362l-3.177,0.554c-0.293-0.67-0.63-1.332-1.015-1.978c-0.387-0.653-0.801-1.258-1.243-1.813 l2.033-2.936c0.346-0.466,0.476-0.981,0.375-1.56c-0.1-0.571-0.399-1.026-0.887-1.375L112.13,54.992L112.13,54.992z M99.48,65.248 c1.16-0.203,2.295-0.176,3.412,0.082c1.11,0.266,2.115,0.71,3.011,1.334c0.891,0.632,1.649,1.427,2.275,2.382 c0.62,0.964,1.03,2.022,1.232,3.183c0.203,1.161,0.175,2.296-0.082,3.412c-0.266,1.111-0.711,2.114-1.336,3.011 c-0.631,0.892-1.425,1.649-2.381,2.275c-0.963,0.62-2.021,1.03-3.182,1.233c-1.161,0.202-2.296,0.175-3.412-0.083 c-1.111-0.266-2.115-0.71-3.011-1.335c-0.892-0.632-1.649-1.426-2.275-2.381c-0.62-0.964-1.03-2.022-1.233-3.183 c-0.202-1.161-0.175-2.296,0.083-3.412c0.266-1.111,0.71-2.115,1.335-3.012c0.631-0.89,1.426-1.648,2.381-2.274 C97.262,65.86,98.319,65.45,99.48,65.248L99.48,65.248z M66.013,13.556c-0.18-0.181-0.348-0.303-0.502-0.368 c-0.146-0.062-0.33-0.092-0.551-0.092c-0.225,0-0.412,0.032-0.563,0.095c-0.16,0.067-0.321,0.183-0.481,0.347l-0.019,0.018v0 l-4.172,4.159c-0.824,0.82-2.099,0.91-3.022,0.272c-0.469-0.301-0.972-0.595-1.504-0.883c-0.58-0.313-1.137-0.59-1.675-0.836 c-0.567-0.259-1.149-0.504-1.743-0.735c-0.519-0.202-1.119-0.415-1.792-0.636c-1.001-0.33-1.637-1.261-1.636-2.261h-0.001V6.238 c0-0.233-0.034-0.433-0.102-0.599c-0.065-0.161-0.171-0.313-0.315-0.459l-0.004,0.004c-0.151-0.15-0.303-0.257-0.455-0.319 c-0.166-0.068-0.366-0.101-0.599-0.101H38.85c-0.212,0-0.392,0.03-0.54,0.092c-0.166,0.068-0.336,0.187-0.511,0.356v0 c-0.142,0.143-0.246,0.291-0.31,0.444c-0.067,0.16-0.101,0.353-0.101,0.58v5.856c0,1.172-0.846,2.146-1.961,2.345 c-0.632,0.151-1.206,0.304-1.714,0.46c-0.575,0.176-1.157,0.378-1.746,0.607c-0.556,0.214-1.125,0.456-1.706,0.724 c-0.583,0.27-1.12,0.538-1.607,0.804c-0.94,0.512-2.075,0.323-2.8-0.392l-0.001,0.001l-4.62-4.553 c-0.019-0.019-0.037-0.037-0.055-0.056l-0.001,0c-0.145-0.153-0.29-0.263-0.435-0.326c-0.133-0.058-0.304-0.088-0.515-0.088 c-0.21,0-0.39,0.033-0.542,0.097c-0.177,0.075-0.351,0.197-0.521,0.363l-5.61,5.623l-0.004,0.004l0.004,0.005 c-0.18,0.18-0.302,0.347-0.367,0.501c-0.062,0.146-0.092,0.33-0.092,0.551c0,0.224,0.031,0.412,0.095,0.563 c0.068,0.16,0.183,0.321,0.347,0.481l0.018,0.019l0,0l4.159,4.172c0.82,0.825,0.91,2.099,0.272,3.023 c-0.301,0.468-0.595,0.971-0.883,1.504c-0.313,0.579-0.59,1.136-0.836,1.674c-0.26,0.568-0.505,1.15-0.735,1.744 c-0.202,0.52-0.415,1.12-0.637,1.792c-0.33,1.001-1.261,1.637-2.261,1.637v0.001H6.239c-0.234,0-0.434,0.034-0.599,0.101 c-0.162,0.066-0.313,0.169-0.454,0.311l-0.009,0.009l0,0c-0.141,0.141-0.245,0.292-0.311,0.454 c-0.067,0.166-0.101,0.366-0.101,0.599v8.028c0,0.211,0.031,0.391,0.092,0.54c0.068,0.165,0.187,0.335,0.357,0.511l0.002,0.002 c0.143,0.141,0.291,0.244,0.442,0.307c0.16,0.066,0.353,0.1,0.581,0.1h5.855c1.172,0,2.146,0.847,2.345,1.962 c0.15,0.629,0.302,1.199,0.458,1.706l0.002-0.001c0.185,0.604,0.387,1.189,0.607,1.755c0.212,0.55,0.457,1.132,0.733,1.745 c0.285,0.63,0.557,1.192,0.814,1.685c0.493,0.944,0.288,2.067-0.43,2.781l0.002,0.001l-4.554,4.541l-0.038,0.037h0.001 c-0.153,0.146-0.263,0.29-0.326,0.436c-0.058,0.133-0.088,0.304-0.088,0.515c0,0.209,0.033,0.39,0.097,0.541 c0.076,0.18,0.201,0.356,0.372,0.53l5.568,5.633c0.165,0.151,0.332,0.261,0.501,0.327c0.171,0.067,0.374,0.102,0.607,0.102 c0.236,0,0.442-0.034,0.618-0.104c0.165-0.065,0.323-0.167,0.472-0.307l4.125-4.19c0.821-0.835,2.108-0.935,3.041-0.291 c0.468,0.3,0.972,0.595,1.505,0.883c0.58,0.313,1.137,0.59,1.674,0.836c0.568,0.26,1.15,0.505,1.744,0.735 c0.519,0.202,1.12,0.415,1.792,0.636c1.001,0.33,1.637,1.262,1.637,2.262h0.001v6.396c0,0.232,0.034,0.433,0.102,0.599 c0.066,0.162,0.169,0.313,0.31,0.454l0.004,0.005l0.005-0.005c0.282,0.282,0.631,0.421,1.053,0.421h8.027 c0.211,0,0.392-0.03,0.54-0.092c0.165-0.068,0.335-0.188,0.511-0.357h0c0.142-0.144,0.245-0.292,0.309-0.444 c0.067-0.16,0.101-0.354,0.101-0.58v-5.856c0-1.173,0.847-2.147,1.961-2.346c0.632-0.15,1.206-0.304,1.714-0.46 c0.575-0.176,1.157-0.378,1.746-0.606c0.55-0.212,1.132-0.457,1.745-0.733c0.63-0.285,1.193-0.557,1.685-0.813 c0.944-0.493,2.067-0.288,2.781,0.43l0.001-0.002l4.541,4.554l0.036,0.038l0.001-0.001c0.147,0.155,0.292,0.265,0.434,0.326 c0.135,0.059,0.315,0.088,0.543,0.088c0.22,0,0.404-0.033,0.554-0.098c0.154-0.066,0.299-0.172,0.436-0.316 c0.03-0.031,0.061-0.062,0.092-0.092l5.598-5.532c0.15-0.164,0.26-0.331,0.326-0.501c0.067-0.171,0.102-0.373,0.102-0.606 c0-0.237-0.035-0.443-0.104-0.618c-0.065-0.166-0.167-0.323-0.306-0.472l-4.191-4.126c-0.835-0.821-0.935-2.109-0.291-3.041 c0.301-0.469,0.595-0.972,0.883-1.504c0.313-0.58,0.59-1.137,0.836-1.676c0.26-0.567,0.505-1.149,0.735-1.743 c0.202-0.519,0.415-1.12,0.637-1.792c0.33-1.002,1.262-1.637,2.262-1.637v-0.001h6.395c0.234,0,0.434-0.033,0.6-0.102 c0.162-0.065,0.313-0.169,0.454-0.31l0.005-0.004l-0.005-0.005c0.282-0.282,0.421-0.632,0.421-1.053V38.85 c0-0.212-0.031-0.392-0.092-0.54c-0.068-0.165-0.188-0.335-0.357-0.511l-0.002-0.003c-0.143-0.141-0.29-0.243-0.442-0.307 c-0.159-0.067-0.353-0.1-0.581-0.1h-5.855c-1.186,0-2.169-0.867-2.352-2.001c-0.133-0.528-0.283-1.067-0.453-1.616 c-0.171-0.55-0.375-1.141-0.617-1.774c-0.015-0.04-0.029-0.08-0.042-0.12c-0.221-0.578-0.445-1.117-0.673-1.616 c-0.253-0.554-0.521-1.09-0.804-1.608c-0.512-0.94-0.322-2.075,0.392-2.8l-0.001,0l4.554-4.62c0.019-0.019,0.037-0.037,0.057-0.055 l-0.001-0.001c0.153-0.145,0.263-0.29,0.326-0.435c0.058-0.133,0.088-0.304,0.088-0.515c0-0.21-0.033-0.39-0.098-0.542 c-0.075-0.176-0.196-0.35-0.363-0.521l-5.623-5.609l-0.005-0.005L66.013,13.556L66.013,13.556z M67.354,8.813 c0.756,0.32,1.432,0.777,2.028,1.373l-0.005,0.005l5.628,5.615l0,0l0.019,0.019c0.592,0.602,1.047,1.269,1.358,2 c0.323,0.76,0.488,1.56,0.488,2.403c0,0.855-0.168,1.659-0.498,2.414c-0.322,0.739-0.789,1.393-1.394,1.969l-3.313,3.36 c0.05,0.105,0.099,0.212,0.147,0.318c0.291,0.639,0.558,1.273,0.799,1.903c0.016,0.036,0.031,0.074,0.046,0.112 c0.245,0.642,0.481,1.335,0.71,2.072l0.076,0.248h4.043c0.853,0,1.652,0.158,2.405,0.473c0.739,0.31,1.404,0.763,1.998,1.36 l0.047,0.047c0.581,0.601,1.021,1.267,1.325,2c0.31,0.75,0.464,1.531,0.464,2.346v8.028c0,1.737-0.599,3.203-1.817,4.421 l-0.004-0.005c-0.608,0.607-1.284,1.062-2.032,1.366c-0.745,0.305-1.539,0.455-2.386,0.455H72.77 c-0.047,0.124-0.096,0.25-0.146,0.377c-0.259,0.667-0.541,1.335-0.847,2.005c-0.319,0.698-0.642,1.353-0.967,1.954 c-0.038,0.071-0.077,0.144-0.118,0.216l2.869,2.823c0.032,0.03,0.063,0.062,0.095,0.094c0.576,0.608,1.01,1.285,1.302,2.026 c0.293,0.743,0.438,1.528,0.438,2.35c0,0.818-0.146,1.598-0.437,2.337c-0.29,0.738-0.718,1.41-1.285,2.02l-0.001-0.001 c-0.023,0.025-0.048,0.051-0.073,0.075l-5.651,5.586c-0.585,0.614-1.247,1.085-1.99,1.405c-0.751,0.323-1.562,0.488-2.434,0.488 c-0.853,0-1.659-0.168-2.422-0.498c-0.756-0.326-1.421-0.797-1.998-1.405l-3.296-3.306c-0.119,0.056-0.236,0.109-0.351,0.161 c-0.611,0.275-1.273,0.552-1.987,0.827c-0.661,0.257-1.356,0.496-2.081,0.718c-0.097,0.03-0.193,0.06-0.289,0.088v4.033 c0,0.852-0.158,1.652-0.473,2.403c-0.31,0.74-0.764,1.405-1.361,1.999l-0.047,0.047c-0.601,0.58-1.267,1.021-1.999,1.324 c-0.75,0.311-1.53,0.465-2.346,0.465H36.85c-1.737,0-3.204-0.6-4.422-1.816l0.004-0.005c-0.607-0.608-1.062-1.284-1.366-2.032 c-0.304-0.745-0.455-1.539-0.455-2.385V72.77c-0.124-0.047-0.249-0.096-0.377-0.145c-0.666-0.26-1.334-0.542-2.003-0.848 c-0.699-0.319-1.353-0.642-1.955-0.967c-0.072-0.039-0.144-0.078-0.215-0.118l-2.824,2.868c-0.03,0.033-0.062,0.064-0.094,0.096 c-0.608,0.576-1.286,1.01-2.026,1.302c-0.743,0.293-1.528,0.438-2.349,0.438c-0.82,0-1.599-0.145-2.338-0.436 c-0.738-0.29-1.41-0.719-2.019-1.286l0.001-0.001c-0.026-0.023-0.051-0.048-0.076-0.073l-5.633-5.697 c-0.592-0.604-1.046-1.27-1.358-2.001c-0.324-0.759-0.488-1.56-0.488-2.402c0-0.855,0.168-1.659,0.498-2.413 c0.325-0.744,0.795-1.401,1.405-1.98l3.306-3.297c-0.056-0.118-0.109-0.235-0.161-0.351c-0.276-0.611-0.552-1.273-0.827-1.986 c-0.267-0.685-0.505-1.376-0.719-2.072l0.002-0.001l-0.002-0.008l-0.087-0.289H6.239c-0.852,0-1.652-0.157-2.404-0.472 c-0.74-0.31-1.405-0.764-1.999-1.36l-0.047-0.048c-0.581-0.601-1.021-1.267-1.324-1.999C0.154,46.473,0,45.691,0,44.877v-8.028 c0-0.846,0.151-1.64,0.455-2.385c0.304-0.745,0.756-1.419,1.36-2.025l0.002-0.002l0.004-0.004l0.004-0.005 c0.606-0.604,1.281-1.058,2.027-1.362s1.54-0.455,2.386-0.455h4.718c0.047-0.124,0.096-0.249,0.146-0.377 c0.259-0.666,0.541-1.334,0.847-2.004c0.319-0.698,0.641-1.352,0.966-1.954c0.042-0.077,0.084-0.155,0.126-0.232l-2.853-2.861 c-0.601-0.595-1.059-1.268-1.374-2.011c-0.325-0.768-0.486-1.571-0.486-2.405c0-0.831,0.161-1.628,0.483-2.393 c0.319-0.756,0.776-1.432,1.373-2.028l0.004,0.004l5.615-5.628l0,0l0.019-0.019c0.602-0.592,1.268-1.047,2-1.359 c0.76-0.323,1.56-0.488,2.403-0.488c0.855,0,1.658,0.168,2.413,0.498c0.739,0.323,1.393,0.789,1.969,1.394l3.355,3.308 c0.102-0.049,0.203-0.096,0.304-0.143c0.617-0.285,1.278-0.563,1.985-0.835c0.661-0.257,1.356-0.497,2.081-0.719 c0.097-0.029,0.193-0.059,0.289-0.087V6.238c0-0.852,0.158-1.652,0.472-2.404c0.31-0.739,0.764-1.405,1.36-1.999l0.048-0.047 c0.601-0.581,1.267-1.021,2-1.324C37.254,0.154,38.035,0,38.85,0h8.028c0.845,0,1.64,0.151,2.385,0.455 c0.759,0.31,1.438,0.763,2.036,1.361l-0.004,0.005l0.004,0.004c0.604,0.603,1.057,1.278,1.362,2.027 c0.304,0.745,0.455,1.54,0.455,2.385v4.719c0.124,0.047,0.25,0.096,0.377,0.145c0.666,0.259,1.334,0.541,2.004,0.847 c0.699,0.32,1.353,0.642,1.954,0.967l0.232,0.126l2.861-2.853c0.596-0.602,1.268-1.059,2.011-1.374 c0.769-0.325,1.571-0.485,2.405-0.485C65.791,8.331,66.589,8.491,67.354,8.813L67.354,8.813z M41.863,23.754 c1.234,0,2.438,0.118,3.616,0.353c1.152,0.23,2.298,0.586,3.438,1.067l0.016,0.007l0.003-0.007 c1.077,0.464,2.103,1.023,3.077,1.675c0.964,0.645,1.855,1.373,2.672,2.183l0.01,0.01v0c0.811,0.818,1.539,1.71,2.184,2.674 c0.652,0.974,1.21,2,1.675,3.076c0.028,0.066,0.054,0.133,0.076,0.2c0.443,1.081,0.773,2.166,0.991,3.256 c0.235,1.177,0.353,2.382,0.353,3.616c0,1.234-0.118,2.439-0.353,3.615c-0.23,1.152-0.586,2.298-1.067,3.439l-0.007,0.016 l0.007,0.002c-0.464,1.076-1.023,2.102-1.675,3.076c-0.645,0.964-1.373,1.855-2.184,2.673l-0.01,0.01l0,0 c-0.817,0.81-1.708,1.538-2.673,2.184c-0.974,0.651-1.999,1.21-3.076,1.675c-0.066,0.028-0.132,0.054-0.2,0.075 c-1.081,0.443-2.166,0.773-3.257,0.991c-1.177,0.235-2.382,0.353-3.616,0.353s-2.438-0.117-3.615-0.353 c-1.152-0.229-2.298-0.585-3.438-1.066l-0.016-0.007l-0.003,0.007c-1.076-0.465-2.102-1.023-3.076-1.675 c-0.964-0.646-1.855-1.374-2.672-2.184l-0.009-0.01l0,0c-0.811-0.817-1.539-1.709-2.184-2.673c-0.652-0.974-1.21-1.999-1.674-3.076 c-0.029-0.065-0.054-0.133-0.076-0.199c-0.443-1.081-0.773-2.167-0.992-3.258c-0.235-1.176-0.353-2.381-0.353-3.615 c0-1.234,0.118-2.438,0.353-3.616c0.23-1.152,0.586-2.298,1.067-3.438l0.006-0.016l-0.006-0.002 c0.464-1.077,1.022-2.102,1.674-3.076c0.646-0.964,1.373-1.855,2.184-2.674l0.009-0.009v0c0.817-0.81,1.708-1.538,2.672-2.183 c0.975-0.652,2-1.211,3.077-1.675c0.067-0.029,0.134-0.054,0.202-0.076c1.08-0.442,2.165-0.773,3.255-0.991 C39.424,23.872,40.629,23.754,41.863,23.754L41.863,23.754z M44.548,28.778c-0.866-0.172-1.761-0.259-2.685-0.259 s-1.819,0.087-2.685,0.259c-0.833,0.166-1.623,0.401-2.37,0.705c-0.045,0.022-0.091,0.044-0.138,0.064 c-0.845,0.365-1.617,0.781-2.314,1.248c-0.705,0.472-1.358,1.006-1.961,1.603c-0.595,0.601-1.129,1.255-1.601,1.96 c-0.466,0.697-0.882,1.469-1.247,2.313l-0.007-0.003c-0.332,0.79-0.586,1.626-0.762,2.511c-0.173,0.866-0.26,1.761-0.26,2.685 s0.086,1.819,0.26,2.685c0.166,0.832,0.401,1.622,0.704,2.369c0.023,0.045,0.044,0.091,0.065,0.139 c0.364,0.845,0.78,1.615,1.247,2.313c0.472,0.705,1.006,1.358,1.603,1.961c0.602,0.596,1.255,1.13,1.96,1.602 c0.697,0.467,1.469,0.883,2.313,1.247l-0.003,0.007c0.789,0.331,1.625,0.586,2.51,0.763c0.866,0.172,1.76,0.259,2.685,0.259 c0.924,0,1.819-0.087,2.685-0.259c0.833-0.166,1.622-0.401,2.369-0.704c0.046-0.023,0.092-0.045,0.139-0.065 c0.845-0.364,1.616-0.78,2.313-1.246c0.705-0.473,1.359-1.007,1.961-1.603c0.596-0.602,1.13-1.256,1.603-1.962 c0.466-0.696,0.882-1.468,1.246-2.312l0.007,0.002c0.332-0.789,0.586-1.625,0.763-2.51c0.173-0.866,0.26-1.761,0.26-2.685 s-0.087-1.819-0.26-2.685c-0.166-0.833-0.401-1.622-0.704-2.369c-0.023-0.045-0.045-0.091-0.065-0.139 c-0.364-0.844-0.78-1.616-1.247-2.313c-0.472-0.706-1.006-1.359-1.601-1.96c-0.602-0.597-1.256-1.131-1.961-1.603 c-0.698-0.467-1.469-0.883-2.314-1.248l0.003-0.007C46.27,29.209,45.434,28.955,44.548,28.778L44.548,28.778z"/></g></svg>',
				"description"          => esc_html__(
					"This is a board specific general setting page. Boards are separate forums in separate pages. By default, there is a one forum board and one setting menu. If you create a second forum board, you'll have a separate set of menus per board. You can create a new board if you want to have more than one forum in your website. For example, you can use boards to create separate forums for different languages \"EN Forum\" and \"FR Forum\". Here, you can enable forum cache and manage some board specific permalinks.",
					"wpforo"
				),
				"description_original" => "This is a board specific general setting page. Boards are separate forums in separate pages. By default, there is a one forum board and one setting menu. If you create a second forum board, you'll have a separate set of menus per board. You can create a new board if you want to have more than one forum in your website. For example, you can use boards to create separate forums for different languages \"EN Forum\" and \"FR Forum\". Here, you can enable forum cache and manage some board specific permalinks.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/board-settings/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/board.php' );
				},
				"options"              => [
					"under_construction" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Under Construction", "wpforo" ),
						"label_original"       => "Under Construction",
						"description"          => esc_html__( "You can enable option Under Construction for each boards separately. (This option has no effect for site admins.)", "wpforo" ),
						"description_original" => "You can enable option Under Construction for each boards separately. (This option has no effect for site admins.)",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/board-settings/#under-construction",
					],
					"cache"              => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Cache", "wpforo" ),
						"label_original"       => "Enable Cache",
						"description"          => esc_html__( "wpForo has own cache system, enable this to reduce sql requests and make your forum run faster.", "wpforo" ),
						"description_original" => "wpForo has own cache system, enable this to reduce sql requests and make your forum run faster.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/board-settings/#enable-cache",
					],
					"url_structure"      => [
						"type"                 => "radio",
						"label"                => esc_html__( "URL Structure", "wpforo" ),
						"label_original"       => "URL Structure",
						"description"          => esc_html__(
							"This option allows managing topic and post URL structure. By default, URLs are built based on topic titles. You can change them to ID and make them shorter.",
							"wpforo"
						),
						"description_original" => "This option allows managing topic and post URL structure. By default, URLs are built based on topic titles. You can change them to ID and make them shorter.",
						"variants"             => [
							[ 'value' => 'short', 'label' => __( 'ID', 'wpforo' ), 'title' => 'https://example.com/postid/123/' ],
							[ 'value' => 'full', 'label' => __( 'Title', 'wpforo' ), 'title' => 'https://example.com/community/sample-forum/sample-topic/' ],
						],
						"docurl"               => "",
					],
				],
			],
			'components'    => [
				"title"                => esc_html__( "Display Components", "wpforo" ),
				"title_original"       => "Display Components",
				"icon"                 => '<svg height="50px" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><defs><style>.cls-9900{fill:none;stroke:#43a6df;stroke-linejoin:round;stroke-width:2px;}</style></defs><title/><g data-name="389-Browser Layout" id="_389-Browser_Layout"><rect class="cls-9900" height="26" width="30" x="1" y="3"/><line class="cls-9900" x1="1" x2="31" y1="9" y2="9"/><line class="cls-9900" x1="4" x2="6" y1="6" y2="6"/><line class="cls-9900" x1="8" x2="10" y1="6" y2="6"/><line class="cls-9900" x1="1" x2="31" y1="15" y2="15"/><line class="cls-9900" x1="8" x2="8" y1="15" y2="29"/></g></svg>',
				"description"          => esc_html__(
					"wpForo is a forum plugin which is loaded in one WordPress page, so the whole forum with its categories, forums, topics, members and profile pages are loaded dynamically in one WordPress page. Almost all components you can see in the page content area (forum area) can be displayed or hidden in wpForo settings, especially here, in the Display Components admin page.",
					"wpforo"
				),
				"description_original" => "wpForo is a forum plugin which is loaded in one WordPress page, so the whole forum with its categories, forums, topics, members and profile pages are loaded dynamically in one WordPress page. Almost all components you can see in the page content area (forum area) can be displayed or hidden in wpForo settings, especially here, in the Display Components admin page.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-components/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/components.php' );
				},
				"options"              => [
					"admin_cp"         => [
						"type"                 => "radio",
						"label"                => esc_html__( "Frontend Admin Control Panel", "wpforo" ),
						"label_original"       => "Frontend Admin Control Panel",
						"description"          => esc_html__(
							"This is a section with some important buttons for quick actions and accesses to certain admin pages. it's located in the forum home page under the forum list. This section is only available for website administrators.",
							"wpforo"
						),
						"description_original" => "This is a section with some important buttons for quick actions and accesses to certain admin pages. it's located in the forum home page under the forum list. This section is only available for website administrators.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-components/#frontend-admin-control-panel",
					],
					"page_title"       => [
						"type"                 => "radio",
						"label"                => esc_html__( "Show Forum Page Title", "wpforo" ),
						"label_original"       => "Show Forum Page Title",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-components/#show-forum-page-title",
					],
					"top_bar"          => [
						"type"                 => "radio",
						"label"                => esc_html__( "Show Forum Menu Bar", "wpforo" ),
						"label_original"       => "Show Forum Menu Bar",
						"description"          => esc_html__( "This menu bar contains the forum menu, notification bell and forum search section.", "wpforo" ),
						"description_original" => "This menu bar contains the forum menu, notification bell and forum search section.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-components/#forum-menu-search-section-and-breadcrumb",
					],
					"top_bar_search"   => [
						"type"                 => "radio",
						"label"                => esc_html__( "Show Search Section in the Forum Menu Bar", "wpforo" ),
						"label_original"       => "Show Search Section in the Forum Menu Bar",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-components/#forum-menu-search-section-and-breadcrumb",
					],
					"sidebar_location" => [
						"type"                 => "radio",
						"label"                => esc_html__( "wpForo Sidebar Location", "wpforo" ),
						"label_original"       => "wpForo Sidebar Location",
						"description"          => esc_html__( "This option allows you to manage the sidebar location in wpForo.", "wpforo" ),
						"description_original" => "This option allows you to manage the sidebar location in wpForo.",
						"variants"             => [
							[ 'value' => 'left', 'label' => __( 'Left', 'wpforo' ), 'title' => 'Always sets the sidebar location to the left, even for Right-to-Left (RTL) layouts.' ],
							[
								'value' => 'default',
								'label' => __( 'Default', 'wpforo' ),
								'title' => 'For Left-to-Right (LTR) displays the right position, and for Right-to-Left (RTL) displays the left position',
							],
							[ 'value' => 'right', 'label' => __( 'Right', 'wpforo' ), 'title' => 'Always sets the sidebar location to the right, even for Right-to-Left (RTL) layouts.' ],
						],
						"docurl"               => "",
					],
					"breadcrumb"       => [
						"type"                 => "radio",
						"label"                => esc_html__( "Show Breadcrumb", "wpforo" ),
						"label_original"       => "Show Breadcrumb",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-components/#forum-menu-search-section-and-breadcrumb",
					],
					"footer"           => [
						"type"                 => "radio",
						"label"                => esc_html__( "Show Forum Footer", "wpforo" ),
						"label_original"       => "Show Forum Footer",
						"description"          => esc_html__(
							"Forum footer is located at the bottom of all forum content. It includes forum statistic, last registered user and post information. As well as it contains the 'Mark all read' button and forums/topics icon description.",
							"wpforo"
						),
						"description_original" => "Forum footer is located at the bottom of all forum content. It includes forum statistic, last registered user and post information. As well as it contains the 'Mark all read' button and forums/topics icon description.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-components/#forum-footer-and-statistics",
					],
					"footer_stat"      => [
						"type"                 => "radio",
						"label"                => esc_html__( "Show Forum Statistic", "wpforo" ),
						"label_original"       => "Show Forum Statistic",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-components/#forum-footer-and-statistics",
					],
					"copyright"        => [
						"type"                 => "radio",
						"label"                => esc_html__( "Help wpForo to grow, show plugin info", "wpforo" ),
						"label_original"       => "Help wpForo to grow, show plugin info",
						"description"          => esc_html__(
							"Please enable this option to help wpForo get more popularity as your thank to the hard work we do for you totally free. This option adds a very small icon in forum footer, which will allow your site visitors recognize the name of forum solution you use.",
							"wpforo"
						),
						"description_original" => "Please enable this option to help wpForo get more popularity as your thank to the hard work we do for you totally free. This option adds a very small icon in forum footer, which will allow your site visitors recognize the name of forum solution you use.",
						"docurl"               => "",
					],
				],
			],
			'forums'        => [
				"title"                => esc_html__( "Display Forums", "wpforo" ),
				"title_original"       => "Display Forums",
				"option"               => 'wpforo_forum_options',
				"icon"                 => '<svg style="enable-background:new 0 0 64 64;" height="60px" version="1.1" viewBox="0 0 64 64" xml:space="preserve" xmlns="http://www.w3.org/2000/svg"><g><rect height="2" width="2" x="5" y="5"/><rect height="2" width="2" x="9" y="5"/><rect height="2" width="2" x="13" y="5"/><path d="M63,15h-6V1H1v48h8v6h22v8h28v-8h4V15z M61,23H11v-6h50V23z M55,3v6H3V3H55z M3,47V11h52v4H9v32H3z M39,41h-8v8H15V31h24   V41z M57,61H33V43h24V61z M59,53V41H41V29H13v22h18v2H11V25h50v28H59z"/><rect height="2" width="2" x="13" y="19"/><rect height="2" width="2" x="17" y="19"/><rect height="2" width="2" x="21" y="19"/><rect height="2" width="20" x="17" y="33"/><rect height="2" width="20" x="17" y="37"/><rect height="2" width="12" x="17" y="41"/><rect height="2" width="12" x="17" y="45"/><rect height="2" width="20" x="35" y="45"/><rect height="2" width="20" x="35" y="49"/><rect height="2" width="20" x="35" y="53"/><rect height="2" width="2" x="53" y="57"/><rect height="2" width="16" x="35" y="57"/></g></svg>',
				"description"          => esc_html__(
					"Here you can manage forum specific components which are displayed on the frontend. All options are grouped by forum layouts. You can find all necessary options to manage forum layout specific features. Each forum layout has its own specific features.",
					"wpforo"
				),
				"description_original" => "Here you can manage forum specific components which are displayed on the frontend. All options are grouped by forum layouts. You can find all necessary options to manage forum layout specific features. Each forum layout has its own specific features.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/forums.php' );
				},
				"options"              => [
					"layout_extended_intro_topics_toggle" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Extended Layout - Recent topics", "wpforo" ),
						"label_original"       => "Extended Layout - Recent topics",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => false, 'label' => __( 'Collapsed', 'wpforo' ) ],
							[ 'value' => true, 'label' => __( 'Expanded', 'wpforo' ) ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#extended-forum-layout",
					],
					"layout_extended_intro_topics_count"  => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Extended Layout - Number of Recent topics", "wpforo" ),
						"label_original"       => "Extended Layout - Number of Recent topics",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#extended-forum-layout",
					],
					"layout_extended_intro_topics_length" => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Extended Layout - Recent topic length", "wpforo" ),
						"label_original"       => "Extended Layout - Recent topic length",
						"description"          => esc_html__( "Set this option value 0 if you want to show the whole title in recent topic area.", "wpforo" ),
						"description_original" => "Set this option value 0 if you want to show the whole title in recent topic area.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#extended-forum-layout",
					],
					"layout_simplified_add_topic_button"  => [
						"type"                 => "radio",
						"label"                => esc_html__( "Simplified Layout - Display Add Topic Button on Forum List", "wpforo" ),
						"label_original"       => "Simplified Layout - Display Add Topic Button on Forum List",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#simplified-forum-layout",
					],
					"layout_qa_intro_topics_toggle"       => [
						"type"                 => "radio",
						"label"                => esc_html__( "Q&A Layout - Recent topics", "wpforo" ),
						"label_original"       => "Q&A Layout - Recent topics",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => false, 'label' => __( 'Collapsed', 'wpforo' ) ],
							[ 'value' => true, 'label' => __( 'Expanded', 'wpforo' ) ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#q-a-forum-layout",
					],
					"layout_qa_intro_topics_count"        => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Q&A Layout - Number of Recent topics", "wpforo" ),
						"label_original"       => "Q&A Layout - Number of Recent topics",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#q-a-forum-layout",
					],
					"layout_qa_intro_topics_length"       => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Q&A Layout - Recent topic length", "wpforo" ),
						"label_original"       => "Q&A Layout - Recent topic length",
						"description"          => esc_html__( "Set this option value 0 if you want to show the whole title in recent topic area.", "wpforo" ),
						"description_original" => "Set this option value 0 if you want to show the whole title in recent topic area.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#q-a-forum-layout",
					],
					"layout_qa_add_topic_button"          => [
						"type"                 => "radio",
						"label"                => esc_html__( "Q&A Layout - Display Add Topic Button on Forum List", "wpforo" ),
						"label_original"       => "Q&A Layout - Display Add Topic Button on Forum List",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#q-a-forum-layout",
					],
					"layout_threaded_intro_topics_toggle" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Threaded Layout - Forums List", "wpforo" ),
						"label_original"       => "Threaded Layout - Forums List",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => false, 'label' => __( 'Collapsed', 'wpforo' ) ],
							[ 'value' => true, 'label' => __( 'Expanded', 'wpforo' ) ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#threaded-forum-layout",
					],
					"layout_threaded_display_subforums"   => [
						"type"                 => "radio",
						"label"                => esc_html__( "Threaded Layout - Display Sub Forums with Parent Forums", "wpforo" ),
						"label_original"       => "Threaded Layout - Display Sub Forums with Parent Forums",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#threaded-forum-layout",
					],
					"layout_threaded_filter_buttons"      => [
						"type"                 => "radio",
						"label"                => esc_html__( "Threaded Layout - Display Thread Filtering Buttons", "wpforo" ),
						"label_original"       => "Threaded Layout - Display Thread Filtering Buttons",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"layout_threaded_add_topic_button"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Threaded Layout - Display Add Topic Button on Forum List", "wpforo" ),
						"label_original"       => "Threaded Layout - Display Add Topic Button on Forum List",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#threaded-forum-layout",
					],
					"layout_threaded_intro_topics_count"  => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Threaded Layout - Number of Recent topics", "wpforo" ),
						"label_original"       => "Threaded Layout - Number of Recent topics",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#threaded-forum-layout",
					],
					"layout_threaded_intro_topics_length" => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Threaded Layout - Recent topic length", "wpforo" ),
						"label_original"       => "Threaded Layout - Recent topic length",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-forums/#threaded-forum-layout",
					],
				
				],
			],
			'styles'        => [
				"title"                => esc_html__( "Colors &amp; Styles", "wpforo" ),
				"title_original"       => "Colors & Styles",
				"icon"                 => '<svg height="58px" data-name="Layer 1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 101.03 122.88"><path d="M42.88,114.05a46.55,46.55,0,0,0,24-3.15,50.83,50.83,0,0,0,9.64-5.19l.64,7.09a56.6,56.6,0,0,1-7.9,4A52.93,52.93,0,0,1,42,120.31,45.8,45.8,0,0,1,15.85,107,55.59,55.59,0,0,1,1.16,79.54,53.4,53.4,0,0,1,6.91,42.23a54.45,54.45,0,0,1,29-24.57,26.55,26.55,0,0,1,12.52-1.74,13.51,13.51,0,0,1,9.4,5.42,3.17,3.17,0,0,1,.33.6c2.93,6.69-.63,10.88-4,14.9-2,2.31-3.84,4.52-2.74,6.64a5,5,0,0,0,3.29,2,20.15,20.15,0,0,0,8,.11c2.6-.42,5.29-1,7.85-1.53,1.86-.39,3.66-.77,5.39-1.09l-1.44,6.72c-.87.17-1.75.36-2.64.55-2.55.54-5.21,1.11-8.17,1.57a26.19,26.19,0,0,1-10.55-.23c-3.41-.89-6-2.66-7.29-5.21-3.07-5.89.14-9.68,3.5-13.64,2.08-2.46,4.26-5,3.2-8a7.45,7.45,0,0,0-5-2.61,20.63,20.63,0,0,0-9.53,1.42A48.13,48.13,0,0,0,12.43,45.32,47,47,0,0,0,7.34,78.23a49.17,49.17,0,0,0,13,24.36,39.44,39.44,0,0,0,22.55,11.46ZM41,101a9.16,9.16,0,1,0,5.25-11.83A9.16,9.16,0,0,0,41,101ZM90.11,0C89.4,11.14,74.8,18,80,29.59a13,13,0,0,0,7.73,6.93,8.55,8.55,0,0,0,4.17.22,8.63,8.63,0,0,0,1.84-.61c7.59-3.44,8.5-12.75,6.25-19.9C97.89,9.64,93.36,3.81,90.11,0Zm9.74,62.1a42.27,42.27,0,0,1-18.21.17l4.93,54.26c.25,3.37.9,5.69,5,6.35a5,5,0,0,0,3.36-1.45,5.47,5.47,0,0,0,1.3-3.55L99.85,62.1ZM94.92,41.2c-2.3.95-6.44,1.45-9.34,0L81.73,59.07c6.39,1.42,12.64,1.69,18.63,0-2.18-8.69-4-8.78-5.44-17.82ZM62.49,81a10.47,10.47,0,0,0,12.35,6.38L73,66.77A10.48,10.48,0,0,0,62.49,81Zm-45.61.86A9.16,9.16,0,1,0,22.13,70a9.15,9.15,0,0,0-5.25,11.84Zm5.37-27a9.09,9.09,0,1,0,5.22-11.75,9.1,9.1,0,0,0-5.22,11.75Z"/></svg>',
				"description"          => esc_html__(
					"Here you can change your forum style. There are six predefined styles (default, red, green, orange, gray and dark), these are set of different colors. You can change them and create your own set of colors. If you need more customization, you can add custom CSS code in the corresponding textarea. Also, you can change font size of forums, topics and post content.",
					"wpforo"
				),
				"description_original" => "Here you can change your forum style. There are six predefined styles (default, red, green, orange, gray and dark), these are set of different colors. You can change them and create your own set of colors. If you need more customization, you can add custom CSS code in the corresponding textarea. Also, you can change font size of forums, topics and post content.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/colors-styles/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/styles.php' );
				},
				"options"              => [
					"font_sizes"             => [
						"type"                 => "wrap",
						"label"                => esc_html__( 'Font Sizes', 'wpforo' ),
						"label_original"       => "Font Sizes:",
						"description"          => __( 'These options allow you to change the font sizes of forum and topic titles as well as the font size of post content.', 'wpforo' ),
						"description_original" => 'These options allow you to change the font sizes of forum and topic titles as well as the font size of post content.',
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/colors-styles/#font-sizes",
						"options_keys"         => [ 'font_size_forum', 'font_size_topic', 'font_size_post_content' ],
					],
					"font_size_forum"        => [
						"type"                 => "select",
						"label"                => esc_html__( "Forums font size:", "wpforo" ),
						"label_original"       => "Forums font size:",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 11, 'label' => '11px' ],
							[ 'value' => 12, 'label' => '12px' ],
							[ 'value' => 13, 'label' => '13px' ],
							[ 'value' => 14, 'label' => '14px' ],
							[ 'value' => 15, 'label' => '15px' ],
							[ 'value' => 16, 'label' => '16px' ],
							[ 'value' => 17, 'label' => '17px' ],
							[ 'value' => 18, 'label' => '18px' ],
							[ 'value' => 19, 'label' => '19px' ],
							[ 'value' => 20, 'label' => '20px' ],
							[ 'value' => 21, 'label' => '21px' ],
							[ 'value' => 22, 'label' => '22px' ],
							[ 'value' => 23, 'label' => '23px' ],
							[ 'value' => 24, 'label' => '24px' ],
							[ 'value' => 25, 'label' => '25px' ],
							[ 'value' => 26, 'label' => '26px' ],
							[ 'value' => 27, 'label' => '27px' ],
						],
					],
					"font_size_topic"        => [
						"type"                 => "select",
						"label"                => esc_html__( "Topics font size:", "wpforo" ),
						"label_original"       => "Topics font size:",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 11, 'label' => '11px' ],
							[ 'value' => 12, 'label' => '12px' ],
							[ 'value' => 13, 'label' => '13px' ],
							[ 'value' => 14, 'label' => '14px' ],
							[ 'value' => 15, 'label' => '15px' ],
							[ 'value' => 16, 'label' => '16px' ],
							[ 'value' => 17, 'label' => '17px' ],
							[ 'value' => 18, 'label' => '18px' ],
							[ 'value' => 19, 'label' => '19px' ],
							[ 'value' => 20, 'label' => '20px' ],
							[ 'value' => 21, 'label' => '21px' ],
							[ 'value' => 22, 'label' => '22px' ],
							[ 'value' => 23, 'label' => '23px' ],
							[ 'value' => 24, 'label' => '24px' ],
							[ 'value' => 25, 'label' => '25px' ],
							[ 'value' => 26, 'label' => '26px' ],
							[ 'value' => 27, 'label' => '27px' ],
						],
					],
					"font_size_post_content" => [
						"type"                 => "select",
						"label"                => esc_html__( "Post content font size:", "wpforo" ),
						"label_original"       => "Post content font size:",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 11, 'label' => '11px' ],
							[ 'value' => 12, 'label' => '12px' ],
							[ 'value' => 13, 'label' => '13px' ],
							[ 'value' => 14, 'label' => '14px' ],
							[ 'value' => 15, 'label' => '15px' ],
							[ 'value' => 16, 'label' => '16px' ],
							[ 'value' => 17, 'label' => '17px' ],
							[ 'value' => 18, 'label' => '18px' ],
							[ 'value' => 19, 'label' => '19px' ],
							[ 'value' => 20, 'label' => '20px' ],
							[ 'value' => 21, 'label' => '21px' ],
							[ 'value' => 22, 'label' => '22px' ],
							[ 'value' => 23, 'label' => '23px' ],
							[ 'value' => 24, 'label' => '24px' ],
							[ 'value' => 25, 'label' => '25px' ],
							[ 'value' => 26, 'label' => '26px' ],
							[ 'value' => 27, 'label' => '27px' ],
						],
					],
					"custom_css"             => [
						"type"                 => "textarea",
						"label"                => esc_html__( "Custom CSS Code", "wpforo" ),
						"label_original"       => "Custom CSS Code",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/colors-styles/#custom-css-code",
					],
					"style"                  => [
						"type"                 => "color",
						"label"                => esc_html__( "Forum Color Styles", "wpforo" ),
						"label_original"       => "Forum Color Styles",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/colors-styles/#forum-styles",
					],
				],
			],
			'topics'        => [
				"title"                => esc_html__( "Display Topics &amp; Posts", "wpforo" ),
				"title_original"       => "Display Topics & Posts",
				"icon"                 => '<svg viewBox="0 0 32 32" height="50px" xmlns="http://www.w3.org/2000/svg"><defs><style>.cls-8512{fill:none;stroke:#43a6df;stroke-linejoin:round;stroke-width:2px;}</style></defs><title/><g data-name="20-List" id="_20-List"><rect class="cls-8512" height="6" width="6" x="1" y="1"/><rect class="cls-8512" height="6" width="6" x="1" y="13"/><rect class="cls-8512" height="6" width="6" x="1" y="25"/><line class="cls-8512" x1="10" x2="32" y1="2" y2="2"/><line class="cls-8512" x1="10" x2="20" y1="6" y2="6"/><line class="cls-8512" x1="10" x2="32" y1="14" y2="14"/><line class="cls-8512" x1="10" x2="20" y1="18" y2="18"/><line class="cls-8512" x1="10" x2="32" y1="26" y2="26"/><line class="cls-8512" x1="10" x2="20" y1="30" y2="30"/></g></svg>',
				"description"          => esc_html__(
					"This is the right place to manage topics and posts displaying options. Here you can manage the number of topics and posts per page, also you can display or hide recent posts under each topic in certain forum layouts. Some options are grouped by forum layouts.",
					"wpforo"
				),
				"description_original" => "This is the right place to manage topics and posts displaying options. Here you can manage the number of topics and posts per page, also you can display or hide recent posts under each topic in certain forum layouts. Some options are grouped by forum layouts.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/topics.php' );
				},
				"options"              => [
					"layout_extended_intro_posts_toggle" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Extended Layout - Recent posts", "wpforo" ),
						"label_original"       => "Extended Layout - Recent posts",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => false, 'label' => __( 'Collapsed', 'wpforo' ) ],
							[ 'value' => true, 'label' => __( 'Expanded', 'wpforo' ) ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#extended-forum-layout",
					],
					"layout_extended_intro_posts_count"  => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Extended Layout - Number of Recent posts", "wpforo" ),
						"label_original"       => "Extended Layout - Number of Recent posts",
						"description"          => esc_html__( "Set this option value 0 if you want to show all posts in recent posts area.", "wpforo" ),
						"description_original" => "Set this option value 0 if you want to show all posts in recent posts area.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#extended-forum-layout",
					],
					"layout_extended_intro_posts_length" => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Extended Layout - Recent post length", "wpforo" ),
						"label_original"       => "Extended Layout - Recent post length",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#extended-forum-layout",
					],
					"layout_qa_posts_per_page"           => [
						"type"                 => "number",
						"min"                  => 5,
						"label"                => esc_html__( "Q&A Layout - Number of Answers per Page", "wpforo" ),
						"label_original"       => "Q&A Layout - Number of Answers per Page",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#q-a-forum-layout",
					],
					"layout_qa_comments_limit_count"     => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Q&A Layout - Number of Comments per Answer", "wpforo" ),
						"label_original"       => "Q&A Layout - Number of Comments per Answer",
						"description"          => esc_html__(
							"This is the number of comments under the answers before clicking the [load more comments] button. Set this option value 0 if you want to show all comments.",
							"wpforo"
						),
						"description_original" => "This is the number of comments under the answers before clicking the [load more comments] button. Set this option value 0 if you want to show all comments",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#q-a-forum-layout",
					],
					"layout_qa_first_post_reply"         => [
						"type"                 => "radio",
						"label"                => esc_html__( "Q&A Layout - Enable Comments on the First Post (question)", "wpforo" ),
						"label_original"       => "Q&A Layout - Enable Comments on the First Post (question)",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#q-a-forum-layout",
					],
					"layout_threaded_posts_per_page"     => [
						"type"                 => "number",
						"min"                  => 5,
						"label"                => esc_html__( "Threaded Layout - Number of Parent Posts per Page", "wpforo" ),
						"label_original"       => "Threaded Layout - Number of Parent Posts per Page",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#threaded-forum-layout",
					],
					"layout_threaded_nesting_level"      => [
						"type"                 => "number",
						"min"                  => 0,
						"max"                  => 7,
						"label"                => esc_html__( "Threaded Layout - Replies Nesting Levels Deep", "wpforo" ),
						"label_original"       => "Threaded Layout - Replies Nesting Levels Deep",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#threaded-forum-layout",
					],
					"layout_threaded_first_post_reply"   => [
						"type"                 => "radio",
						"label"                => esc_html__( "Threaded Layout - First Post Reply Button", "wpforo" ),
						"label_original"       => "Threaded Layout - First Post Reply Button",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#threaded-forum-layout",
					],
					"include_subforums_topics"           => [
						"type"                 => "radio",
						"label"                => esc_html__( "Display Subforums' Topics with the Parent Forum Topics", "wpforo" ),
						"label_original"       => "Display Subforums' Topics with the Parent Forum Topics",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"topics_per_page"                    => [
						"type"                 => "number",
						"min"                  => 5,
						"label"                => esc_html__( "Number of Topics per Page", "wpforo" ),
						"label_original"       => "Number of Topics per Page",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#number-of-topics-and-posts-per-page",
					],
					"posts_per_page"                     => [
						"type"                 => "number",
						"min"                  => 5,
						"label"                => esc_html__( "Number of Posts per Page", "wpforo" ),
						"label_original"       => "Number of Posts per Page",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#number-of-topics-and-posts-per-page",
					],
					"search_max_results"                 => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Maximum Number of Search Results", "wpforo" ),
						"label_original"       => "Maximum number of search results",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#maximum-number-of-search-results",
					],
					"union_first_post"                   => [
						"type"                 => "radio",
						"label"                => esc_html__( "Stick Topic's First Post on Top for Certain Forum Layout", "wpforo" ),
						"label_original"       => "Stick Topic's First Post on Top for Certain Forum Layout",
						"description"          => esc_html__(
							"This option keeps the first topic post on top when you navigate through pages of that topic. You can manage this option by forum layouts.",
							"wpforo"
						),
						"description_original" => "This option keeps the first topic post on top when you navigate through pages of that topic. You can manage this option by forum layouts.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#maximum-number-of-search-results",
					],
					"recent_posts_type"                  => [
						"type"                 => "radio",
						"label"                => esc_html__( "Recent Posts Display Type", "wpforo" ),
						"label_original"       => "Recent Posts Display Type",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'topics', 'label' => 'Topics' ],
							[ 'value' => 'posts', 'label' => 'Posts' ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-topics-posts/#recent-posts-display-type",
					],
					"topic_head"                         => [
						"type"                 => "radio",
						"label"                => esc_html__( "Display Topic Statistics and Overview Panel", "wpforo" ),
						"label_original"       => "Display Topic Statistics and Overview Panel",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"topic_head_expanded"                => [
						"type"                 => "radio",
						"label"                => esc_html__( "Keep Expanded Topic Overview in the Statistic Panel", "wpforo" ),
						"label_original"       => "Keep Expanded Topic Overview in the Statistic Panel",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
				],
			],
			'tags'          => [
				"title"                => esc_html__( "Topic Tags", "wpforo" ),
				"title_original"       => "Topic Tags",
				"icon"                 => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" width="122.879px" height="50px" viewBox="0 0 122.879 106.533" enable-background="new 0 0 122.879 106.533" xml:space="preserve"><g><path d="M79.704,105.239c-1.2,1.072-3.042,0.971-4.115-0.229c-1.073-1.2-0.971-3.042,0.229-4.115l39.432-35.204l0.012-0.01v-0.001 c0.569-0.503,0.995-1.045,1.281-1.629c0.281-0.574,0.443-1.229,0.487-1.963c0.044-0.732-0.041-1.404-0.255-2.016 c-0.213-0.605-0.572-1.192-1.077-1.761l-0.048-0.055L70.895,9.569c-1.092-1.187-1.015-3.035,0.172-4.126s3.034-1.015,4.126,0.172 l44.758,48.693c0.038,0.039,0.075,0.078,0.112,0.119c1.004,1.127,1.744,2.37,2.22,3.727c0.474,1.352,0.665,2.777,0.575,4.277 c-0.09,1.497-0.448,2.89-1.081,4.182c-0.629,1.283-1.512,2.427-2.653,3.435l-0.001-0.001L79.704,105.239L79.704,105.239z M54.383,3.131h0.738v0.006c0.788,0,1.574,0.317,2.149,0.942l44.758,48.694c0.038,0.039,0.075,0.078,0.111,0.119 c1.005,1.128,1.745,2.37,2.221,3.727c0.474,1.352,0.665,2.777,0.575,4.278c-0.091,1.496-0.449,2.89-1.082,4.181 c-0.628,1.283-1.511,2.427-2.653,3.435v-0.001l-39.291,35.077c-0.026,0.028-0.055,0.055-0.083,0.082 c-1.113,1.032-2.346,1.787-3.698,2.266c-1.34,0.474-2.774,0.666-4.302,0.575c-1.53-0.091-2.936-0.451-4.219-1.076 c-1.287-0.628-2.435-1.511-3.447-2.647l-42.083-47.2l0.001-0.001c-0.505-0.567-0.79-1.33-0.732-2.147l0.086-1.181L0.006,3.114h0 c-0.008-0.124-0.009-0.25,0-0.377c0.101-1.609,1.487-2.833,3.097-2.731L53.451,3.22c0.063-0.01,0.127-0.02,0.192-0.028 C53.934,3.151,54.182,3.131,54.383,3.131L54.383,3.131z M53.922,9.068c-0.195,0.019-0.389,0.018-0.579-0.002L6.066,6.047 l3.208,46.027c0.009,0.135,0.009,0.269-0.001,0.4h0.001l-0.011,0.148l41.261,46.28c0.498,0.559,1.045,0.984,1.642,1.275 c0.6,0.292,1.267,0.462,2.001,0.505c0.738,0.044,1.41-0.042,2.016-0.256c0.594-0.21,1.149-0.554,1.663-1.031 c0.059-0.055,0.12-0.106,0.182-0.155l39.296-35.082l0.012-0.011l-0.001-0.001c0.569-0.502,0.996-1.045,1.281-1.628 c0.282-0.575,0.443-1.229,0.488-1.964c0.044-0.732-0.042-1.404-0.256-2.016c-0.213-0.605-0.571-1.192-1.077-1.761l-0.048-0.055 L53.922,9.068L53.922,9.068z M24.268,12.234c1.474,0.04,2.833,0.335,4.078,0.885c1.256,0.555,2.392,1.36,3.406,2.413l0.013,0.015 c1.006,1.042,1.766,2.204,2.278,3.485c0.519,1.298,0.779,2.68,0.779,4.145c0,0.127-0.008,0.252-0.024,0.375 c-0.071,1.347-0.37,2.605-0.895,3.772l-0.005,0.01l0.005,0.002c-0.563,1.251-1.38,2.386-2.446,3.404 c-0.063,0.06-0.128,0.117-0.195,0.17c-1.023,0.923-2.145,1.621-3.363,2.09c-1.271,0.489-2.614,0.72-4.031,0.69v0.01 c-1.463,0-2.838-0.278-4.123-0.832c-1.265-0.544-2.419-1.354-3.459-2.424l-0.012-0.012v0c-1.028-1.067-1.796-2.246-2.304-3.537 c-0.504-1.285-0.738-2.663-0.698-4.133c0.039-1.495,0.334-2.873,0.892-4.135c0.564-1.278,1.387-2.43,2.473-3.457 c1.058-1,2.232-1.75,3.521-2.247S22.818,12.195,24.268,12.234L24.268,12.234z M25.991,18.468c-0.57-0.251-1.19-0.387-1.86-0.405 c-0.681-0.019-1.305,0.082-1.87,0.3c-0.568,0.22-1.11,0.573-1.625,1.059c-0.5,0.472-0.874,0.992-1.124,1.56 c-0.257,0.582-0.394,1.222-0.412,1.918c-0.019,0.674,0.085,1.297,0.31,1.869c0.224,0.571,0.582,1.11,1.069,1.617l0,0 c0.485,0.497,1.01,0.867,1.574,1.109c0.543,0.234,1.148,0.351,1.814,0.351v0.011l0.069,0.001c0.688,0.019,1.315-0.082,1.882-0.3 c0.528-0.203,1.032-0.518,1.512-0.943c0.033-0.036,0.068-0.07,0.104-0.104c0.506-0.483,0.888-1.009,1.143-1.576l0.004,0.002 c0.224-0.502,0.355-1.047,0.392-1.636c-0.002-0.041-0.003-0.083-0.003-0.125c0-0.74-0.115-1.398-0.345-1.973 c-0.231-0.579-0.586-1.115-1.063-1.607l-0.018-0.018C27.072,19.087,26.554,18.717,25.991,18.468L25.991,18.468z M52.422,31.36 c-1.092-1.187-1.015-3.035,0.171-4.126c1.187-1.092,3.034-1.015,4.126,0.172l24.931,27.04c1.092,1.187,1.015,3.034-0.172,4.126 s-3.034,1.016-4.126-0.172L52.422,31.36L52.422,31.36z M40.49,42.661c-1.092-1.187-1.015-3.034,0.171-4.126 c1.187-1.092,3.034-1.016,4.126,0.171l24.931,27.04c1.092,1.187,1.015,3.034-0.172,4.126c-1.188,1.092-3.035,1.016-4.127-0.171 L40.49,42.661L40.49,42.661z M28.074,53.962c-1.092-1.187-1.015-3.035,0.171-4.126c1.187-1.092,3.035-1.015,4.126,0.171 l24.931,27.04c1.092,1.188,1.016,3.034-0.171,4.126s-3.034,1.016-4.126-0.171L28.074,53.962L28.074,53.962z"/></g></svg>',
				"description"          => esc_html__(
					"Tags are a free form method of categorization, currently being used to group and categorize similar topics. The goal of using these tags in the forum is to create a \"heat map\", an index of topics covered within the Forum. The larger the size of the tag, the more discussions about the topic.",
					"wpforo"
				),
				"description_original" => "Tags are a free form method of categorization, currently being used to group and categorize similar topics. The goal of using these tags in the forum is to create a \"heat map\", an index of topics covered within the Forum. The larger the size of the tag, the more discussions about the topic.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/topic-tags/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/tags.php' );
				},
				"options"              => [
					"max_per_topic" => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Maximum Number of Tags per Topic", "wpforo" ),
						"label_original"       => "Maximum Number of Tags per Topic",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/topic-tags/#maximum-number-of-tags-per-topic",
					],
					"per_page"      => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Number of Tags per Page", "wpforo" ),
						"label_original"       => "Number of Tags per Page",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/topic-tags/#number-of-tags-per-page",
					],
					"length"        => [
						"type"                 => "number",
						"min"                  => 1,
						"max"                  => 50,
						"label"                => esc_html__( "Maximum Tag Length", "wpforo" ),
						"label_original"       => "Maximum Tag Length",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/topic-tags/#maximum-tags-length",
					],
					"suggest_limit" => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Maximum Number of Suggested Tags", "wpforo" ),
						"label_original"       => "Maximum Number of Suggested Tags",
						"description"          => esc_html__( "Tags are suggested while you type in the tag field. Using this option you can limit the maximum number of suggested tags.", "wpforo" ),
						"description_original" => "Tags are suggested while you type in the tag field. Using this option you can limit the maximum number of suggested tags.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/topic-tags/#maximum-number-of-suggested-tags",
					],
					"lowercase"     => [
						"type"                 => "radio",
						"label"                => esc_html__( "Force Lowercase Tags", "wpforo" ),
						"label_original"       => "Force Lowercase Tags",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/topic-tags/#force-lowercase-tags",
					],
				],
			],
			'posting'       => [
				"title"                => esc_html__( "Posting &amp; Editor Settings", "wpforo" ),
				"title_original"       => "Posting & Editor Settings",
				"icon"                 => '<svg version="1.1" height="53px" xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 122.88 122.88" style="enable-background:new 0 0 122.88 122.88" xml:space="preserve"><style type="text/css">.st0{fill-rule:evenodd;clip-rule:evenodd;}</style><g><path class="st0" d="M79.7,31.87c-0.7-0.65-1.5-1-2.4-0.95c-0.9,0-1.7,0.35-2.35,1.05l-5.29,5.49L84.49,51.8l5.34-5.59 c0.65-0.65,0.9-1.5,0.9-2.4c0-0.9-0.35-1.75-1-2.35L79.7,31.87L79.7,31.87L79.7,31.87z M12.51,0h97.85c3.44,0,6.57,1.41,8.84,3.67 c2.27,2.27,3.67,5.4,3.67,8.84v97.85c0,3.44-1.41,6.57-3.67,8.84c-2.27,2.27-5.4,3.67-8.84,3.67H12.51c-3.44,0-6.57-1.41-8.84-3.67 c-2.27-2.27-3.67-5.4-3.67-8.84V12.51c0-3.44,1.41-6.57,3.67-8.84C5.94,1.41,9.07,0,12.51,0L12.51,0z M110.37,5.39H12.51 c-1.96,0-3.74,0.8-5.03,2.1c-1.29,1.29-2.1,3.08-2.1,5.03v97.85c0,1.96,0.8,3.74,2.1,5.03c1.29,1.29,3.08,2.1,5.03,2.1h97.85 c1.96,0,3.74-0.8,5.03-2.1c1.29-1.29,2.1-3.08,2.1-5.03V12.51c0-1.96-0.8-3.74-2.1-5.03C114.1,6.19,112.32,5.39,110.37,5.39 L110.37,5.39z M51.93,85.61c-1.95,0.65-3.95,1.25-5.89,1.9c-1.95,0.65-3.9,1.3-5.89,1.95c-4.64,1.5-7.19,2.35-7.74,2.5 c-0.55,0.15-0.2-2,0.95-6.49l3.7-14.13l0.3-0.32L51.93,85.61L51.93,85.61L51.93,85.61L51.93,85.61z M42.74,65.41l22.9-23.78 l14.83,14.28L57.33,79.99L42.74,65.41L42.74,65.41z"/></g></svg>',
				"description"          => esc_html__(
					"Here, you can find all settings related to topic and post creation including rich editor settings, text limitations, attachments control, option to enable WordPress plugin shortcodes in wpForo posts, post drafting and preview options. Some settings are grouped by forum layouts and some of them are general for all forum layouts.",
					"wpforo"
				),
				"description_original" => "Here, you can find all settings related to topic and post creation including rich editor settings, text limitations, attachments control, option to enable WordPress plugin shortcodes in wpForo posts, post drafting and preview options. Some settings are grouped by forum layouts and some of them are general for all forum layouts.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/posting.php' );
				},
				"options"              => [
					"qa_display_answer_editor"      => [
						"type"                 => "radio",
						"label"                => esc_html__( "Q&A Layout - Display Answer Editor", "wpforo" ),
						"label_original"       => "Q&A Layout - Display Answer Editor",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#q-a-layout-%E2%80%93-display-answer-editor",
					],
					"qa_comments_rich_editor"       => [
						"type"                 => "radio",
						"label"                => esc_html__( "Q&A Layout - Comment Form Type", "wpforo" ),
						"label_original"       => "Q&A Layout - Comment Form Type",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => false, 'label' => __( 'Text Editor', 'wpforo' ) ],
							[ 'value' => true, 'label' => __( 'Visual Editor', 'wpforo' ) ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#q-a-layout-%E2%80%93-comment-form-type",
					],
					"threaded_reply_rich_editor"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Threaded Layout - Reply Form Type", "wpforo" ),
						"label_original"       => "Threaded Layout - Reply Form Type",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => false, 'label' => __( 'Text Editor', 'wpforo' ) ],
							[ 'value' => true, 'label' => __( 'Visual Editor', 'wpforo' ) ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#threaded-layout-%E2%80%93-reply-form-type",
					],
					"topic_title_min_length"        => [
						"type"                 => "number",
						"min"                  => 1,
						"label"                => esc_html__( "Topic Title Minimum Length", "wpforo" ),
						"label_original"       => "Topic Title Minimum Length",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"topic_title_max_length"        => [
						"type"                 => "number",
						"min"                  => 0,
						"max"                  => 250,
						"label"                => esc_html__( "Topic Title Maximum Length", "wpforo" ),
						"label_original"       => "Topic Title Maximum Length",
						"description"          => esc_html__( "Topic title length cannot be set more than 250 character.", "wpforo" ),
						"description_original" => "Topic title length cannot be set more than 250 character.",
						"docurl"               => "",
					],
					"topic_body_min_length"         => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Topic Content Minimum Length", "wpforo" ),
						"label_original"       => "Topic Content Minimum Length",
						"description"          => esc_html__( "Set this option value 0 to remove this limit.", "wpforo" ),
						"description_original" => "Set this option value 0 to remove this limit.",
						"docurl"               => "",
					],
					"topic_body_max_length"         => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Topic Content Maximum Length", "wpforo" ),
						"label_original"       => "Topic Content Maximum Length",
						"description"          => esc_html__( "Set this option value 0 to remove this limit.", "wpforo" ),
						"description_original" => "Set this option value 0 to remove this limit.",
						"docurl"               => "",
					],
					"post_body_min_length"          => [
						"type"                 => "number",
						"min"                  => 1,
						"label"                => esc_html__( "Reply Content Minimum Length", "wpforo" ),
						"label_original"       => "Reply Content Minimum Length",
						"description"          => esc_html__( "You cannot set this value 0.", "wpforo" ),
						"description_original" => "You cannot set this value 0.",
						"docurl"               => "",
					],
					"post_body_max_length"          => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Reply Content Maximum Length", "wpforo" ),
						"label_original"       => "Reply Content Maximum Length",
						"description"          => esc_html__( "Set this option value 0 to remove this limit.", "wpforo" ),
						"description_original" => "Set this option value 0 to remove this limit.",
						"docurl"               => "",
					],
					"comment_body_min_length"       => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Comment Content Minimum Length", "wpforo" ),
						"label_original"       => "Comment Content Minimum Length",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"comment_body_max_length"       => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Comment Content Maximum Length", "wpforo" ),
						"label_original"       => "Comment Content Maximum Length",
						"description"          => esc_html__( "Set this option value 0 to remove this limit.", "wpforo" ),
						"description_original" => "Set this option value 0 to remove this limit.",
						"docurl"               => "",
					],
					"edit_own_topic_durr"           => [
						"type"                 => "minute",
						"min"                  => 0,
						"label"                => esc_html__( "Allow Edit Own Topic within (minutes)", "wpforo" ),
						"label_original"       => "Allow Edit Own Topic within (minutes)",
						"description"          => esc_html__( "Set this option value 0 to remove time limit.", "wpforo" ),
						"description_original" => "Set this option value 0 to remove time limit.",
						"docurl"               => "",
					],
					"delete_own_topic_durr"         => [
						"type"                 => "minute",
						"min"                  => 0,
						"label"                => esc_html__( "Allow Delete Own Topic within (minutes)", "wpforo" ),
						"label_original"       => "Allow Delete Own Topic within (minutes)",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"edit_own_post_durr"            => [
						"type"                 => "minute",
						"min"                  => 0,
						"label"                => esc_html__( "Allow Edit Own Post within (minutes)", "wpforo" ),
						"label_original"       => "Allow Edit Own Post within (minutes)",
						"description"          => esc_html__( "Set this option value 0 to remove time limit.", "wpforo" ),
						"description_original" => "Set this option value 0 to remove time limit.",
						"docurl"               => "",
					],
					"delete_own_post_durr"          => [
						"type"                 => "minute",
						"min"                  => 0,
						"label"                => esc_html__( "Allow Delete Own Post within (minutes)", "wpforo" ),
						"label_original"       => "Allow Delete Own Post within (minutes)",
						"description"          => esc_html__( "Set this option value 0 to remove time limit.", "wpforo" ),
						"description_original" => "Set this option value 0 to remove time limit.",
						"docurl"               => "",
					],
					"edit_topic"                    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Display Topic Editing Information", "wpforo" ),
						"label_original"       => "Display Topic Editing Information",
						"description"          => esc_html__(
							"The topic edit logging information 'This post was modified 2 hours ago by John' is displayed under modified topic first post content...",
							"wpforo"
						),
						"description_original" => "The topic edit logging information 'This post was modified 2 hours ago by John' is displayed under modified topic first post content...",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#display-topic-and-post-editing-information",
					],
					"edit_post"                     => [
						"type"                 => "radio",
						"label"                => esc_html__( "Display Post Editing Information", "wpforo" ),
						"label_original"       => "Display Post Editing Information",
						"description"          => esc_html__( "The post edit logging information 'This post was modified 2 hours ago by John' is displayed under modified post content...", "wpforo" ),
						"description_original" => "The post edit logging information 'This post was modified 2 hours ago by John' is displayed under modified post content...",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#display-topic-and-post-editing-information",
					],
					"edit_log_display_limit"        => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Limit Post Editing Information", "wpforo" ),
						"label_original"       => "Limit Post Editing Information",
						"description"          => esc_html__(
							"Limit the post edit logging information 'This post was modified 2 hours ago by John'. Set this option value 0 if you want to disable limiting.",
							"wpforo"
						),
						"description_original" => "Limit the post edit logging information 'This post was modified 2 hours ago by John'. Set this option value 0 if you want to disable limiting.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#limit-post-editing-information",
					],
					"is_preview_on"                 => [
						"type"                 => "radio",
						"label"                => esc_html__( "Post Preview", "wpforo" ),
						"label_original"       => "Post Preview",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#post-preview-and-auto-drafting",
					],
					"is_draft_on"                   => [
						"type"                 => "radio",
						"label"                => esc_html__( "Post Revisions and Draft Saving", "wpforo" ),
						"label_original"       => "Post Revisions and Draft Saving",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#post-preview-and-auto-drafting",
					],
					"auto_draft_interval"           => [
						"type"                 => "select",
						"label"                => esc_html__( "Auto Drafting Interval", "wpforo" ),
						"label_original"       => "Auto Drafting Interval",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 0, 'label' => __( 'disabled', 'wpforo' ) ],
							[ 'value' => 15000, 'label' => __( '15 seconds', 'wpforo' ) ],
							[ 'value' => 30000, 'label' => __( '30 seconds', 'wpforo' ) ],
							[ 'value' => 60000, 'label' => __( '1 minute', 'wpforo' ) ],
							[ 'value' => 120000, 'label' => __( '2 minute', 'wpforo' ) ],
							[ 'value' => 300000, 'label' => __( '5 minute', 'wpforo' ) ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#post-preview-and-auto-drafting",
					],
					"max_drafts_per_page"           => [
						"type"                 => "number",
						"min"                  => 3,
						"label"                => esc_html__( "Max Number of Revisions", "wpforo" ),
						"label_original"       => "Max Number of Revisions",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#post-preview-and-auto-drafting",
					],
					"max_upload_size"               => [
						"type"                 => "megabyte",
						"label"                => esc_html__( "Maximum upload file size", "wpforo" ),
						"label_original"       => "Maximum upload file size",
						"description"          => esc_html__(
							"You can not set this value more than 'upload_max_filesize' and 'post_max_size'. If you want to increase server parameters please contact to your hosting service support.",
							"wpforo"
						),
						"description_original" => "You can not set this value more than 'upload_max_filesize' and 'post_max_size'. If you want to increase server parameters please contact to your hosting service support.",
						"docurl"               => "",
					],
					"attachs_to_medialib"           => [
						"type"                 => "radio",
						"label"                => esc_html__( "Insert Forum Attachments to Media Library", "wpforo" ),
						"label_original"       => "Insert Forum Attachments to Media Library",
						"description"          => esc_html__( "Enable this option to be able manage forum attachments in Dashboard > Media > Library admin page.", "wpforo" ),
						"description_original" => "Enable this option to be able manage forum attachments in Dashboard > Media > Library admin page.",
						"docurl"               => "",
					],
					"topic_editor_toolbar_location" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Toolbar Location of the Topic Editor", "wpforo" ),
						"label_original"       => "Toolbar Location of the Topic Editor",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'top', 'label' => __( 'Top', 'wpforo' ) ],
							[ 'value' => 'bottom', 'label' => __( 'Bottom', 'wpforo' ) ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#rich-editor-toolbar-location",
					],
					"reply_editor_toolbar_location" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Toolbar Location of the Reply Editor", "wpforo" ),
						"label_original"       => "Toolbar Location of the Reply Editor",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'top', 'label' => __( 'Top', 'wpforo' ) ],
							[ 'value' => 'bottom', 'label' => __( 'Bottom', 'wpforo' ) ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/posting-editor-settings/#rich-editor-toolbar-location",
					],
					"content_do_shortcode"          => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable WordPress Shortcodes in Post Content", "wpforo" ),
						"label_original"       => "Enable WordPress Shortcodes in Post Content",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					'extra_html_tags'               => [
						"type"                 => "textarea",
						"label"                => esc_html__( "Allow extra HTML tags", "wpforo" ),
						"label_original"       => "Allow extra HTML tags",
						"description"          => __(
							'By default, wpForo allows all secure HTML tags in post content. Allowing a new HTML tag may affect your forum security. For example the &lt;iframe&gt; and &lt;script&gt; HTML tags may be used by spammers and hackers to load 3rd party ads and viruses to forum. <br>Example of adding a new HTML tags: <code>b, em, p, code, style, a(href title), img(src alt title), embed(src width height) ...</code>',
							"wpforo"
						),
						"description_original" => 'By default, wpForo allows all secure HTML tags in post content. Allowing a new HTML tag may affect your forum security. For example the &lt;iframe&gt; and &lt;script&gt; HTML tags may be used by spammers and hackers to load 3rd party ads and viruses to forum. <br>Example of adding a new HTML tags: <code>b, em, p, code, style, a(href title), img(src alt title), embed(src width height) ...</code>',
						"docurl"               => "",
					],
				],
			],
			'members'       => [
				"title"                => esc_html__( "Display Members", "wpforo" ),
				"title_original"       => "Display Members",
				"icon"                 => '<svg height="50px" data-name="Layer 1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 122.88 119.86"><title>publisher</title><path d="M20.72,72a3,3,0,0,1-2.84-3.1,3,3,0,0,1,2.84-3.1H57.47a3,3,0,0,1,2.84,3.1A3,3,0,0,1,57.47,72ZM83.08,95.75c-1-1.53-2.77-3.62-2.77-5.42a2.92,2.92,0,0,1,1.94-2.64c-.09-1.51-.15-3.06-.15-4.59,0-.9,0-1.82.05-2.72A6.52,6.52,0,0,1,82.46,79a9.7,9.7,0,0,1,4.32-5.48,12.28,12.28,0,0,1,2.34-1.12c1.48-.54.76-2.88,2.39-2.91,3.79-.1,10,3.22,12.47,5.86a8.84,8.84,0,0,1,2.49,5.93L106.32,88a2.17,2.17,0,0,1,1.59,1.37c.52,2.1-1.66,4.71-2.67,6.38s-4.5,5.74-4.51,5.78a1.39,1.39,0,0,0,.32.77c5.54,7.62,21.83,1.74,21.83,16.89H65.33c0-15.16,16.29-9.27,21.82-16.89a1.68,1.68,0,0,0,.4-.79c0-.1-4.1-5.12-4.47-5.71Zm8-76.89h10.18A7.16,7.16,0,0,1,106.39,21a7.26,7.26,0,0,1,2.13,5.13V61.9l-6.27-2.46V26.13a1,1,0,0,0-1-1H91V57.88l-6.24,2.46V7.27a1,1,0,0,0-1-1H7.24a1,1,0,0,0-1,1V93.72a1,1,0,0,0,1,1H64.42L62,101H23.66v11.6a1,1,0,0,0,1,1H56.37l-2.46,6.24H24.73a7.31,7.31,0,0,1-7.27-7.28V101H7.27A7.31,7.31,0,0,1,0,93.72V7.27A7.16,7.16,0,0,1,2.14,2.14,7.23,7.23,0,0,1,7.27,0H83.79a7.18,7.18,0,0,1,5.14,2.14,7.27,7.27,0,0,1,2.14,5.13V18.86Zm-70.38,10a3,3,0,0,1-2.85-3.1,3,3,0,0,1,2.85-3.1H69.77a3,3,0,0,1,2.84,3.1,3,3,0,0,1-2.84,3.1Zm0,21.57a3,3,0,0,1-2.85-3.1,3,3,0,0,1,2.85-3.1H69.77a3,3,0,0,1,2.84,3.1,3,3,0,0,1-2.84,3.1Z"/></svg>',
				"description"          => esc_html__(
					"This is the members' directory setting page. wpForo forum has a members' page where you can see all members. The members' page is called members directory, and it's general for all forum boards. Starting from wpForo 2.0 you can create multiple separate forum boards in your website, however the members' directory will be one for all boards.",
					"wpforo"
				),
				"description_original" => "This is the members' directory setting page. wpForo forum has a members' page where you can see all members. The members' page is called members directory, and it's general for all forum boards. Starting from wpForo 2.0 you can create multiple separate forum boards in your website, however the members' directory will be one for all forums.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-members/",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/members.php' );
				},
				"options"              => [
					"list_order"       => [
						"type"                 => "select",
						"label"                => esc_html__( "Members List Ordering", "wpforo" ),
						"label_original"       => "Members List Ordering",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'online_time', 'label' => __( 'Last Online time', 'wpforo' ) ],
							[ 'value' => 'posts__asc', 'label' => __( 'Posts Ascending', 'wpforo' ) ],
							[ 'value' => 'posts__desc', 'label' => __( 'Posts Descending', 'wpforo' ) ],
							[ 'value' => 'user_registered__asc', 'label' => __( 'Join Date Ascending', 'wpforo' ) ],
							[ 'value' => 'user_registered__desc', 'label' => __( 'Join Date Descending', 'wpforo' ) ],
							[ 'value' => 'display_name__asc', 'label' => __( 'Display Name A->Z', 'wpforo' ) ],
							[ 'value' => 'display_name__desc', 'label' => __( 'Display Name Z->A', 'wpforo' ) ],
						],
						"docurl"               => "",
					],
					"hide_inactive"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Hide Inactive Members", "wpforo" ),
						"label_original"       => "Hide Inactive Members",
						"description"          => esc_html__( "Members who have not confirmed their email after the registration are set as 'inactive'. In most cases they are spam bots.", "wpforo" ),
						"description_original" => "Members who have not confirmed their email after the registration are set as 'inactive'. In most cases they are spam bots.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/display-members/#hide-inactive-members",
					],
					"members_per_page" => [
						"type"                 => "number",
						"min"                  => 5,
						"label"                => esc_html__( "Number of Members per Page", "wpforo" ),
						"label_original"       => "Number of Members per Page",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"search_type"      => [
						"type"                 => "radio",
						"label"                => esc_html__( "Members Search Type", "wpforo" ),
						"label_original"       => "Members Search Type",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'search', 'label' => __( 'Search', 'wpforo' ) ],
							[ 'value' => 'filter', 'label' => __( 'Filter', 'wpforo' ) ],
						],
						"docurl"               => "",
					],
				],
			],
			'profiles'      => [
				"title"                => esc_html__( "Member Profile", "wpforo" ),
				"title_original"       => "Member Profile",
				"icon"                 => '<svg height="50px" data-name="Layer 1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 115.34 122.88"><path d="M57.34,116.35a3.27,3.27,0,0,1,0,6.53H7.6a7.61,7.61,0,0,1-7.6-7.6V7.6A7.61,7.61,0,0,1,7.6,0H98.36A7.61,7.61,0,0,1,106,7.6v54a10.65,10.65,0,0,1-.09,1.43h-.66l0,0h-.42l-.09,0,0,0h0l-.09,0-.08,0,0,0h0l-.08,0h0l-.09,0,0,0h0l-.08,0h0l-.09,0,0,0h0l-.08,0h0l-.08,0h-.08l-.09,0h0l-.08,0,0,0h0l-.08,0h0l-.08,0h-.09l-.08,0h0l-.08,0h0l0,0-.08,0h0l-.08,0h-.34l-.07,0h0l-.08,0h-.08l-.08,0H99.43V7.6a1.09,1.09,0,0,0-1.07-1.07H7.6a1.11,1.11,0,0,0-.76.31,1.13,1.13,0,0,0-.31.76V115.27a1,1,0,0,0,.32.75,1.12,1.12,0,0,0,.76.32Zm35.05.13L76,122.08l.13-18.25,16.29,12.65ZM80.73,97.72l17.68-24a1.6,1.6,0,0,1,1.91-.56L114.8,84a1.38,1.38,0,0,1,.15,2.06L97,110.38,80.73,97.72ZM29.13,62.84a6.51,6.51,0,0,1-.2-3.36c.46-3.55,1.38-4.21,4.66-5.12,3.68-1,7.84-1.09,11-3.27a10.9,10.9,0,0,0,.74-1.43c.37-.85.71-1.77.92-2.4a30.53,30.53,0,0,1-2.42-3.45l-2.45-3.9A7.17,7.17,0,0,1,40,36.35a2.85,2.85,0,0,1,.25-1.28,2.31,2.31,0,0,1,.84-1,1.93,1.93,0,0,1,.59-.3,62.62,62.62,0,0,1-.12-7,9.88,9.88,0,0,1,.3-1.59A9.43,9.43,0,0,1,46,19.93a13,13,0,0,1,3.47-1.54c.78-.22-.67-2.71.14-2.79,3.89-.4,10.18,3.15,12.9,6.09a9.47,9.47,0,0,1,2.39,6l-.15,6.36h0A1.74,1.74,0,0,1,66,35.38a5.44,5.44,0,0,1-.67,3.34h0l0,.09-2.8,4.59a23.76,23.76,0,0,1-3.42,4.72l.38.54A16.81,16.81,0,0,0,61.27,51a.16.16,0,0,1,.06.08c2.36,1.66,7.2,2.4,10.79,3.34,3,.79,4.1,1,4.77,4.28.34,1.62.35,3.4-.09,4.13ZM26.67,97.89a3.31,3.31,0,0,1,0-6.61H71l0,0,0,0h0l0,0v0l0,0,0,0h0l-.06.09h0v0l0,0v0l0,0h0l0,.05h0l0,0v0l0,0v0l0,0h0l0,0h0l0,0v0l0,0h0l0,0h0l0,0h0l0,0v0l0,0v0l0,0h0l0,.05h0l0,0v0l0,0h0l0,0h0l0,0h0l0,0v0l0,0h0l0,0h0l0,0v0l0,0V93h0l0,0h0l0,0v.13h0l0,0h0l0,0h0v0l0,0v0l0,0h0l0,0h0l0,0v.08h0l0,0h0l0,0h0V94h0l0,0h0v.12h0v0l-.07,0-.06,0h0l-.06,0-.06,0h0l0,0-.06,0,0,0,0,0-.06,0,0,0h0l-.06,0,0,0h0l-.07.05-.06,0h0l-.06,0-.06.06h0l-.05,0-.06,0,0,0,0,0-.06.05,0,0,0,0-.06,0,0,0h0l-.06.06-.05,0h0l-.06.06,0,0h0l0,0-.06.06h0l0,0-.06.06,0,0,0,0,0,.06,0,0,0,0,0,.06,0,.05h0l0,.06,0,0h0l-.06.06,0,.06h0l-.05,0,0,.06v0l0,0,0,.06,0,0,0,0-.05.06,0,0,0,0,0,.06,0,0h0l0,.07,0,.06h0l0,.07,0,.06h0l-.05.06,0,.06v0l0,0,0,.07,0,0,0,0,0,.07,0,0,0,0,0,.07,0,0h0l0,.07,0,.07h0l0,.07Zm1.27-18.16a3.27,3.27,0,1,1,0-6.53H75.78a3.27,3.27,0,1,1,0,6.53Z"/></svg>',
				"description"          => esc_html__(
					"wpForo forum has a powerful user profile system which can be managed in this setting page. Here you can hide/show forum header and footer on profile pages, manage profile components like avatars, member titles, nickname, URL structure, signature etc... In case you use BuddyPress or Ultimate Members plugins, you can switch forum profile page to the plugin profile page.",
					"wpforo"
				),
				"description_original" => "wpForo forum has a powerful user profile system which can be managed in this setting page. Here you can hide/show forum header and footer on profile pages, manage profile components like avatars, member titles, nickname, URL structure, signature etc... In case you use BuddyPress or Ultimate Members plugins, you can switch forum profile page to the plugin profile page.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-profile/",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/profiles.php' );
				},
				"options"              => [
					"profile"                  => [
						"type"                 => "select",
						"label"                => esc_html__( "Profile Page", "wpforo" ),
						"label_original"       => "Profile Page",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[
								'value' => 'default',
								'label' => esc_html__( 'Default', 'wpforo' ),
								'title' => esc_attr__( 'wpForo Profile only for the forum, WordPress Profile for others, including comment authors, post authors and etc...', 'wpforo' ),
							],
							[
								'value' => 'wpforo',
								'label' => esc_html__( 'wpForo', 'wpforo' ),
								'title' => esc_attr__( 'wpForo Profile for the whole website including forum, comment authors, post authors and etc...', 'wpforo' ),
							],
							[
								'value'    => 'bp',
								'label'    => esc_html__( 'BuddyPress', 'wpforo' ),
								'title'    => esc_attr__( 'BuddyPress Profile for the whole website including forum.', 'wpforo' ),
								'disabled' => ! class_exists( 'BP_Component' ),
							],
							[
								'value'    => 'um',
								'label'    => esc_html__( 'Ultimate Member', 'wpforo' ),
								'title'    => esc_attr__( 'Ultimate Member Profile for the whole website including forum.', 'wpforo' ),
								'disabled' => ! function_exists( 'UM' ),
							],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-profile/#profile-page",
					],
					"profile_header"           => [
						"type"                 => "radio",
						"label"                => esc_html__( "Show Forum Header on Profile Pages", "wpforo" ),
						"label_original"       => "Show Forum Header on Profile Pages",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-profile/#show-forum-header-and-footer-on-profile-pages",
					],
					"profile_footer"           => [
						"type"                 => "radio",
						"label"                => esc_html__( "Show Forum Footer under Profile Pages", "wpforo" ),
						"label_original"       => "Show Forum Footer under Profile Pages",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-profile/#show-forum-header-and-footer-on-profile-pages",
					],
					"url_structure"            => [
						"type"                 => "radio",
						"label"                => esc_html__( "Members URL Structure", "wpforo" ),
						"label_original"       => "Members URL Structure",
						"description"          => esc_html__( "By default ", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-profile/#members-url-structure-and-member-nickname",
						"variants"             => [
							[ 'value' => 'id', 'label' => 'USER_ID', 'title' => wpforo_url( 'profile', 'member' ) . 'USER_ID/' ],
							[ 'value' => 'nicename', 'label' => 'USER_NICENAME', 'title' => wpforo_url( 'profile', 'member' ) . 'USER_NICENAME/' ],
						],
					],
					"online_status_timeout"    => [
						"type"                 => "minute",
						"label"                => esc_html__( "Online Status Timeout (minutes)", "wpforo" ),
						"label_original"       => "Online Status Timeout (minutes)",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"custom_title_is_on"       => [
						"type"                 => "radio",
						"label"                => esc_html__( "Member Custom Titles", "wpforo" ),
						"label_original"       => "Member Custom Titles",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-profile/#member-custom-title",
					],
					"default_title"            => [
						"type"                 => "text",
						"label"                => esc_html__( "Member Custom Title by Default", "wpforo" ),
						"label_original"       => "Member Custom Title by Default",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-profile/#member-custom-title",
					],
					"title_groupids"           => [
						"type"                 => "checkbox",
						"label"                => esc_html__( "Display Member Primary Usergroup", "wpforo" ),
						"label_original"       => "Display Member Primary Usergroup",
						"description"          => esc_html__( "The names of selected usergroups will be displayed under the corresponding usergroup's users avatar in forum posts.", "wpforo" ),
						"description_original" => "The selected usergroup names will be displayed under the corresponding usergroup users avatar in forum posts.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-profile/#display-primary-and-secondary-usergroups-under-post-author-avata",
						"variants"             => $this->get_variants_usergroups(),
					],
					"title_secondary_groupids" => [
						"type"                 => "checkbox",
						"label"                => esc_html__( "Display Member Secondary Usergroup", "wpforo" ),
						"label_original"       => "Display Member Secondary Usergroup",
						"description"          => esc_html__(
							"The names of selected secondary usergroups will be displayed under the corresponding usergroup's users avatar in forum posts.",
							"wpforo"
						),
						"description_original" => "The names of selected secondary usergroups will be displayed under the corresponding usergroup's users avatar in forum posts.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-profile/#display-primary-and-secondary-usergroups-under-post-author-avata",
						"variants"             => $this->get_variants_usergroups( [], true ),
					],
					"mention_nicknames"        => [
						"type"                 => "radio",
						"label"                => esc_html__( "Display Member Nicknames", "wpforo" ),
						"label_original"       => "Display Member Nicknames",
						"description"          => esc_html__(
							"Nicknames are displayed with '@' symbol, like @john and are used for the user mentioning in posts. Please note, that this is not the user login name (username). If they are the same, users can change their nickname in the profile account page to secure the login name.",
							"wpforo"
						),
						"description_original" => "Nicknames are displayed with '@' symbol, like @john and are used for the user mentioning in posts. Please note, that this is not the user login name (username). If they are the same, users can change their nickname in the profile account page to secure the login name.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-profile/#members-url-structure-and-member-nickname",
					],
					"avatars"                  => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Avatars", "wpforo" ),
						"label_original"       => "Enable Avatars",
						"description"          => esc_html__(
							"Using this option you can disable user avatars for the forum but you cannot show avatars if the WordPress native avatar option is disabled. So please make sure the 'Show Avatars' option is enabled in WordPress Settings > Discussion admin page.",
							"wpforo"
						),
						"description_original" => "Using this option you can disable user avatars for the forum but you cannot show avatars if the WordPress native avatar option is disabled. So please make sure the 'Show Avatars' option is enabled in WordPress Settings > Discussion admin page.",
						"docurl"               => "",
					],
					"custom_avatars"           => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Custom Avatars", "wpforo" ),
						"label_original"       => "Enable Custom Avatars",
						"description"          => esc_html__(
							"By default wpForo uses WordPress avatars. In most cases it's the Gravatar. If this option is enabled, users will be able to set their avatars with external URLs or upload from their devices.",
							"wpforo"
						),
						"description_original" => "By default wpForo uses WordPress avatars. In most cases it's the Gravatar. If this option is enabled, users will be able to set their avatars with external URLs or upload from their devices.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-profile/#enable-custom-avatars",
					],
					"replace_avatar"           => [
						"type"                 => "radio",
						"label"                => esc_html__( "Site-wide Usage of Forum Profile Avatar", "wpforo" ),
						"label_original"       => "Site-wide Usage of Forum Profile Avatar",
						"description"          => esc_html__( "If this option is enabled, wpForo will try to use its avatar in all available places.", "wpforo" ),
						"description_original" => "If this option is enabled, wpForo will try to use its avatar in all available places.",
						"docurl"               => "",
					],
					"default_cover"            => [
						"type"                 => "cover",
						"label"                => esc_html__( "The Default Profile Cover Image", "wpforo" ),
						"label_original"       => "The Default Profile Cover Image",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"signature"                => [
						"type"                 => "radio",
						"label"                => esc_html__( "Allow Member Signature", "wpforo" ),
						"label_original"       => "Allow Member Signature",
						"description"          => esc_html__( "You can edit usergroups and enable/disable the signature for each usergroups individually.", "wpforo" ),
						"description_original" => "You can edit usergroups and enable/disable the signature for each usergroups individually.",
						"docurl"               => "",
					],
				],
			],
			'rating'        => [
				"title"                => esc_html__( "Member Rating", "wpforo" ),
				"title_original"       => "Member Rating",
				"icon"                 => '<svg version="1.1" height="50px" xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 122.88 90.37" style="enable-background:new 0 0 122.88 90.37" xml:space="preserve"><g><path d="M25.75,69.03c0,1.68-1.36,3.03-3.03,3.03c-1.68,0-3.03-1.36-3.03-3.03c0-11.53,4.67-21.97,12.23-29.53 c7.56-7.56,18-12.23,29.53-12.23c11.53,0,21.97,4.67,29.53,12.23c7.56,7.56,12.23,18,12.23,29.53c0,1.68-1.36,3.03-3.03,3.03 c-1.67,0-3.03-1.36-3.03-3.03c0-9.86-3.99-18.78-10.45-25.24c-6.46-6.46-15.38-10.45-25.24-10.45c-9.86,0-18.78,4-25.24,10.45 C29.74,50.25,25.75,59.18,25.75,69.03L25.75,69.03z M87.82,90.37c-0.16-4.78,0.11-4.85-5.48-6.94c-3.28-1.23-7.47-2.79-10.48-4.66 c-3.54,3.28-5.81,7.03-5.89,11.6H87.82L87.82,90.37z M35.06,90.37c0.16-4.78-0.11-4.85,5.48-6.94c3.28-1.23,7.47-2.79,10.48-4.66 c3.54,3.28,5.81,7.03,5.89,11.6H35.06L35.06,90.37z M54.32,74.26c0.15-1.21-3.43-5.8-4.08-8.01c-1.4-2.22-1.9-5.76-0.37-8.11 c0.61-0.93,0.35-2.6,0.35-3.88c0-12.79,22.41-12.8,22.41,0c0,1.62-0.37,2.85,0.51,4.12c1.47,2.13,0.71,5.9-0.53,7.88 c-0.79,2.31-4.54,6.69-4.28,8.01C68.55,80.83,54.27,80.61,54.32,74.26L54.32,74.26L54.32,74.26z M122.88,35.28l-5.14,6.64 l3.65,7.56l-7.9-2.84l-6.06,5.81l0.26-8.39l-7.4-3.97l8.06-2.35l1.49-8.27l4.72,6.94L122.88,35.28L122.88,35.28L122.88,35.28 L122.88,35.28z M96.78,9.17l-1.13,8.32l6.94,4.73l-8.26,1.5l-2.35,8.06l-3.97-7.4l-8.39,0.26l5.81-6.06l-2.85-7.91l7.56,3.65 L96.78,9.17L96.78,9.17L96.78,9.17L96.78,9.17z M0,35.28l8.32,1.13l4.73-6.94l1.5,8.26l8.06,2.35l-7.4,3.97l0.26,8.39L9.4,46.63 l-7.91,2.84l3.65-7.56L0,35.28L0,35.28L0,35.28L0,35.28z M26.1,9.17l6.64,5.14l7.56-3.65l-2.83,7.9l5.81,6.06l-8.39-0.26l-3.97,7.4 l-2.35-8.06l-8.27-1.49l6.94-4.73L26.1,9.17L26.1,9.17L26.1,9.17L26.1,9.17z M61.44,0l3.18,7.77l8.37,0.62l-6.4,5.43l2,8.16 l-7.14-4.42l-7.14,4.42l2-8.16l-6.42-5.43l8.37-0.62L61.44,0L61.44,0L61.44,0L61.44,0z"/></g></svg>',
				"description"          => esc_html__(
					                          "wpForo has built-in member reputation system based on points user get when he/she creates new topic, posts, when gets likes or dislikes. This system grants Rating Titles and Rating Badges. All component of this system (number of posts, titles, badges) can be customized and changed in this admin section.",
					                          'wpforo'
				                          ) . '<br><br>' . esc_html__(
					                          " For an advanced user ranking and members' point awarding system we recommend using myCRED plugin with the integration addon: ",
					                          "wpforo"
				                          ) . '<a href="https://gvectors.com/product/wpforo-mycred/" target="_blank">wpForo myCRED Integration.</a>',
				"description_original" => "wpForo has built-in member reputation system based on points user get when he/she creates new topic, posts, when gets likes or dislikes. This system grants Rating Titles and Rating Badges. All component of this system (number of posts, titles, badges) can be customized and changed in this admin section.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-rating/",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/rating.php' );
				},
				"options"              => [
					"rating"          => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Member Rating Badges", "wpforo" ),
						"label_original"       => "Enable Member Rating Badges",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-rating/#enable-member-rating-badges-and-titles",
					],
					"rating_title"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Member Rating Titles", "wpforo" ),
						"label_original"       => "Enable Member Rating Titles",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-rating/#enable-member-rating-badges-and-titles",
					],
					"member_rating"   => [
						"label"                => esc_html__( "Member Reputation and Titles", "wpforo" ),
						"label_original"       => "Member Reputation and Titles",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"topic_points"    => [
						"type"                 => "float",
						"min"                  => 0,
						"label"                => esc_html__( "Points for One Topic", "wpforo" ),
						"label_original"       => "Points for One Topic",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-rating/#reputation-point-counting",
					],
					"post_points"     => [
						"type"                 => "float",
						"min"                  => 0,
						"label"                => esc_html__( "Points for One Post", "wpforo" ),
						"label_original"       => "Points for 1 Post",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-rating/#reputation-point-counting",
					],
					"like_points"     => [
						"type"                 => "float",
						"min"                  => 0,
						"label"                => esc_html__( "Points for One Like", "wpforo" ),
						"label_original"       => "Points for One Post",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-rating/#reputation-point-counting",
					],
					"dislike_points"  => [
						"type"                 => "float",
						"max"                  => 0,
						"label"                => esc_html__( "Points for One Dislike", "wpforo" ),
						"label_original"       => "Points for One Dislike",
						"description"          => esc_html__( "This should be set a negative value like '-1' to decrease the user reputation points if he/she receives dislikes.", "wpforo" ),
						"description_original" => "This should be set a negative value like '-1' to decrease the user reputation points if he/she receives dislikes.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/member-rating/#reputation-point-counting",
					],
					"rating_title_ug" => [
						"type"                 => "checkbox",
						"label"                => esc_html__( "Enable Reputation Titles for Selected Usergroups", "wpforo" ),
						"label_original"       => "Enable Reputation Titles for Selected Usergroups",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
						"variants"             => $this->get_variants_usergroups(),
					],
					"rating_badge_ug" => [
						"type"                 => "checkbox",
						"label"                => esc_html__( "Enable Reputation Badges for Selected Usergroups", "wpforo" ),
						"label_original"       => "Enable Reputation Badges for Selected Usergroups",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
						"variants"             => $this->get_variants_usergroups(),
					],
					"levels"          => [
						"type"                 => "ratinglevels",
						"label"                => esc_html__( "Member Reputation and Titles", "wpforo" ),
						"label_original"       => "Member Reputation and Titles",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
				],
			],
			'authorization' => [
				"title"                => esc_html__( "Login &amp; Registration", "wpforo" ),
				"title_original"       => "Login & Registration",
				"icon"                 => '<svg version="1.1" height="50px" xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 122.879 88.057" enable-background="new 0 0 122.879 88.057" xml:space="preserve"><g><path d="M60.033,55.557c-0.61,0.332-1.37,0.107-1.697-0.499c-0.332-0.609-0.107-1.37,0.502-1.696 c2.88-1.563,4.221-3.584,4.491-5.561c0.144-1.055-0.021-2.114-0.417-3.097c-0.405-0.993-1.047-1.897-1.86-2.625 c-1.584-1.415-3.805-2.16-6.125-1.583c-0.941,0.233-1.754,0.703-2.467,1.367c-0.757,0.703-1.411,1.624-1.984,2.711 c-0.323,0.614-1.08,0.848-1.689,0.528c-0.614-0.323-0.847-1.08-0.528-1.689c0.703-1.334,1.526-2.479,2.499-3.387 c1.019-0.945,2.201-1.624,3.575-1.964c3.191-0.79,6.23,0.217,8.389,2.147c1.098,0.981,1.969,2.206,2.517,3.555 c0.552,1.363,0.777,2.859,0.568,4.376C65.437,50.85,63.702,53.564,60.033,55.557L60.033,55.557L60.033,55.557z M95.493,78.362 h0.854c0.775,0,1.409-0.633,1.409-1.409v-2.288c0-0.775-0.634-1.407-1.409-1.407h-5.098c-0.775,0-1.408,0.632-1.408,1.407v2.288 c0,0.776,0.633,1.409,1.408,1.409h0.867l-1.662,8.493h6.643L95.493,78.362L95.493,78.362L95.493,78.362z M9.851,88.039V50.395 c-2.002,0.771-3.876,0.787-5.433,0.258c-1.216-0.412-2.242-1.147-2.995-2.097c-0.754-0.95-1.232-2.105-1.378-3.38 c-0.222-1.977,0.368-4.21,2.054-6.306l0,0c0.085-0.102,0.179-0.205,0.291-0.291L51.208,0.472c0.633-0.582,1.609-0.641,2.311-0.094 l32.55,25.305l-0.305,0.104c-4.906,1.702-9.154,4.79-11.706,9.272c-9.218-7.405-19.367-15.417-21.31-16.928 c-3.927,2.986-38.107,29.303-38.107,30.415v39.511L9.851,88.039L9.851,88.039L9.851,88.039z M80.221,3.039L94.91,3.638v19.841 l-14.689-9.694V3.039L80.221,3.039L80.221,3.039L80.221,3.039z M64.717,86.855c0.658-8.566-1.022-8.194,6.154-10.887 c3.579-1.343,8.154-3.044,11.447-5.091l6.146,15.978H64.717L64.717,86.855L64.717,86.855z M86.582,65.963 c0.156-1.317-3.742-6.337-4.452-8.741c-1.526-2.433-2.067-6.285-0.404-8.854c0.664-1.017,0.378-2.835,0.378-4.236 c0-13.964,24.465-13.971,24.465,0c0,1.766-0.404,3.108,0.554,4.497c1.604,2.321,0.776,6.435-0.573,8.593 c-0.866,2.528-4.961,7.308-4.674,8.741C102.116,73.134,86.536,72.893,86.582,65.963L86.582,65.963L86.582,65.963z M105.564,70.142 c3.006,1.923,7.269,3.5,10.678,4.641c6.694,2.236,6.676,2.543,6.63,12.073H99.209L105.564,70.142L105.564,70.142L105.564,70.142z M50.437,65.444c-4.744-6.373-4.86-12.411,0.9-18.023l-4.148-2.291c-3.322-1.292-9.416,8.234-7.061,10.779l3.232,4.794 l-0.667,1.391c-0.184,0.413-0.004,0.703,0.446,0.904l0.438,0.118l-8.406,16.637l1.701,3.628l3.404-1.002l0.876-1.701l-0.839-1.571 l1.849-0.396l0.43-0.835l-0.966-1.473l0.634-1.231l1.845-0.232l0.601-1.165l-1.031-1.348l0.659-1.275l1.881-0.307l2.561-4.889 l0.434,0.237C49.962,66.651,50.351,66.356,50.437,65.444L50.437,65.444L50.437,65.444z M59.367,69.44l-1.252,4.672L56.376,74.8 l-0.364,1.361l1.268,1.076l-0.331,1.239l-1.718,0.605l-0.352,1.309l1.231,1.207l-0.237,0.888l-1.69,0.765l1.129,1.33l-0.487,1.812 l-3.047,1.665l-2.381-3.121l4.541-16.947l-1.1-0.295c-0.224-0.061-0.364-0.294-0.302-0.523l0.462-1.722 c-3.329-2.524-4.979-6.893-3.833-11.175c1.473-5.503,7.028-8.824,12.534-7.589c0.025,0.295,0.021,0.593-0.02,0.892 c-0.103,0.744-0.426,1.51-1.023,2.25c-0.081,0.099-0.167,0.2-0.257,0.299c-1.759-0.114-3.407,1.022-3.881,2.79 c-0.205,0.761-0.16,1.529,0.081,2.221c0.058,0.246,0.152,0.483,0.275,0.716c0.29,0.532,0.716,0.937,1.214,1.194 c0.328,0.218,0.696,0.38,1.093,0.491c1.976,0.527,4-0.609,4.589-2.553c0.597-0.527,1.117-1.084,1.563-1.657 c0.867-1.117,1.46-2.303,1.812-3.509c2.39,2.577,3.441,6.287,2.464,9.936c-1.146,4.282-4.767,7.245-8.906,7.765l-0.462,1.721 c-0.061,0.227-0.295,0.364-0.523,0.304L59.367,69.44L59.367,69.44L59.367,69.44z"/></g></svg>',
				"description"          => esc_html__(
					                          "wpForo is one of the very rare plugins which has its own user registration and login pages. Both features are based on WordPress functions. Here you can control user registration, account approval and deleting settings. Also, you can manage redirections after user login, register and password resting actions.",
					                          "wpforo"
				                          ) . '<br><br>' . esc_html__(
					                          'If you need to customize the registration form and add custom fields, you should check out user custom field builder addon:',
					                          'wpforo'
				                          ) . ' <a href="https://gvectors.com/product/wpforo-user-custom-fields/" target="_blank">wpForo Users Custom Field.</a>',
				"description_original" => "wpForo is one of the very rare plugins which has its own user registration and login pages. Both features are based on WordPress functions. Here you can control user registration, account approval and deleting settings. Also, you can manage redirections after user login, register and password resting actions.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/login-registration/",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/authorization.php' );
				},
				"options"              => [
					"user_register"                     => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable User Registration", "wpforo" ),
						"label_original"       => "Enable User Registration",
						"description"          => esc_html__(
							"This option is not synced with WordPress 'Anyone can register' option in Dashboard > Settings > General admin page. If this option is enabled new users will always be able to register.",
							"wpforo"
						),
						"description_original" => "This option is not synced with WordPress 'Anyone can register' option in Dashboard > Settings > General admin page. If this option is enabled new users will always be able to register.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/login-registration/#enable-user-registration",
					],
					"user_register_email_confirm"       => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable User Registration email confirmation", "wpforo" ),
						"label_original"       => "Enable User Registration email confirmation",
						"description"          => esc_html__( "If you have enabled this option, after registering, user can not login without confirming the email.", "wpforo" ),
						"description_original" => "If you have enabled this option, after registering, user can not login without confirming the email.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/login-registration/#enable-user-registration-email-confirmation",
					],
					"manually_approval"                 => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable User Manually Approval System", "wpforo" ),
						"label_original"       => "Enable User Manually Approval System",
						"description"          => esc_html__( "If you have enabled this option, after registering, user can not login without manually approving of administrator.", "wpforo" ),
						"description_original" => "If you have enabled this option, after registering, user can not login without manually approving of administrator.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/login-registration/#enable-user-manually-approval-system",
					],
					"manually_approval_contact_form"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Admin Contacting Form under the \"Awaiting approval\" Message", "wpforo" ),
						"label_original"       => "Admin Contacting Form under the \"Awaiting approval\" Message",
						"description"          => esc_html__(
							"This option displays a contact form under the 'Awaiting approval' text to allow inactive users send a message to website administrator. This message will be sent to the administrator email address.",
							"wpforo"
						),
						"description_original" => "This option displays a contact form under the 'Awaiting approval' text to allow inactive users send a message to website administrator. This message will be sent to the administrator email address.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/login-registration/#enable-user-manually-approval-system",
					],
					"role_synch"                        => [
						"type"                 => "radio",
						"label"                => esc_html__( "Role-Usergroup Synchronization", "wpforo" ),
						"label_original"       => "Role-Usergroup Synchronization",
						"description"          => esc_html__(
							"Keep enabled this option to synch WordPress User Roles with Forum Usergroups. This connection allows to automatically change Usergroup of a user when his/her User Role is changed by administrators or by membership plugins. In other words this option allows to manage Usergroups based on Users Roles, thus you can directly control users forum accesses based on Users Roles. If this option is turned off, User Roles don't have any affection to users forum accesses, they are only controlled by forum Usergroups.",
							"wpforo"
						),
						"description_original" => "Keep enabled this option to synch WordPress User Roles with Forum Usergroups. This connection allows to automatically change Usergroup of a user when his/her User Role is changed by administrators or by membership plugins. In other words this option allows to manage Usergroups based on Users Roles, thus you can directly control users forum accesses based on Users Roles. If this option is turned off, User Roles don't have any affection to users forum accesses, they are only controlled by forum Usergroups.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/login-registration/#role-usergroup-synchronization",
					],
					"user_delete_method"                => [
						"type"                 => "radio",
						"label"                => esc_html__( "User Delete Method", "wpforo" ),
						"label_original"       => "User Delete Method",
						"description"          => esc_html__(
							"Choose which method to use when user delete action comes. [Soft] option will not remove any content this user has already created. It'll only remove the user with all account information. The  posts' author name will be changed to \"Anonymous ID\". [Hard] option will remove the user data and any content this user has already created.",
							"wpforo"
						),
						"description_original" => "Choose which method to use when user delete action comes. [Soft] option will not remove any content this user has already created. It'll only remove the user with all account information. The  posts' author name will be changed to \"Anonymous ID\". [Hard] option will remove the user data and any content this user has already created.",
						"variants"             => [
							[
								'value' => 'soft',
								'label' => __( 'Soft', 'wpforo' ),
								'title' => __( 'This method will delete only profile information so all content which created this user will be kept and renamed as Anonymous.', 'wpforo' ),
							],
							[ 'value' => 'hard', 'label' => __( 'Hard', 'wpforo' ), 'title' => __( 'This method will also remove all user information and content.', 'wpforo' ) ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/login-registration/#user-delete-method",
					],
					"send_email_after_user_delete"      => [
						"type"                 => "radio",
						"label"                => esc_html__( "Send Email to Admins After User Self Delete", "wpforo" ),
						"label_original"       => "Send Email to Admins After User Self Delete",
						"description"          => "",
						"description_original" => "",
						"docurl"               => "",
					],
					"use_our_register_url"              => [
						"type"                 => "radio",
						"label"                => esc_html__( "Replace Registration Page URL to Forum Registration Page URL", "wpforo" ),
						"label_original"       => "Replace Registration Page URL to Forum Registration Page URL",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"use_our_login_url"                 => [
						"type"                 => "radio",
						"label"                => esc_html__( "Replace Login Page URL to Forum Login Page URL", "wpforo" ),
						"label_original"       => "Replace Login Page URL to Forum Login Page URL",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"use_our_lostpassword_url"          => [
						"type"                 => "radio",
						"label"                => esc_html__( "Replace Reset Password Page URL to Forum Reset Password Page URL", "wpforo" ),
						"label_original"       => "Replace Reset Password Page URL to Forum Reset Password Page URL",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"custom_auth_urls"                  => [
						"type"                 => "wrap",
						"label"                => esc_html__( 'Custom Authorization URLs', 'wpforo' ),
						"label_original"       => "Custom Authorization URLs",
						"description"          => __(
							'Use this option only if you have set other pages for authorization. wpForo doesn\'t change its own URLs, these options are only for other plugin compatibility. For example, if you use BuddyPress or Ultimate Member plugin you can set your profile plugin url',
							'wpforo'
						),
						"description_original" => "Use this option only if you have set other pages for authorization. wpForo doesn\'t change its own URLs, these options are only for other plugin compatibility. For example, if you use BuddyPress or Ultimate Member plugin you can set your profile plugin url",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/login-registration/#custom-authorization-urls",
						"options_keys"         => [ 'login_url', 'register_url', 'lost_password_url' ],
					],
					"login_url"                         => [
						"type"                 => "text",
						"placeholder"          => esc_attr__( "Login URL", 'wpforo' ),
						"label"                => "",
						"label_original"       => "",
						"description"          => "",
						"description_original" => "",
					],
					"register_url"                      => [
						"type"                 => "text",
						"placeholder"          => esc_attr__( "Register URL", 'wpforo' ),
						"label"                => "",
						"label_original"       => "",
						"description"          => '',
						"description_original" => '',
					],
					"lost_password_url"                 => [
						"type"                 => "text",
						"placeholder"          => esc_attr__( "Lost Password", 'wpforo' ),
						"label"                => "",
						"label_original"       => "",
						"description"          => '',
						"description_original" => '',
					],
					"custom_redirect_urls"              => [
						"type"                 => "wrap",
						"label"                => esc_html__( 'Custom Redirection URLs after following actions:', 'wpforo' ),
						"label_original"       => "Custom Redirection URLs after following actions:",
						"description"          => esc_html__( 'For member profile, account and subscription pages use following URLs:', 'wpforo' ) . '<br>' . wpforo_url(
								'profile',
								'member'
							) . '<br>' . wpforo_url( 'account', 'member' ) . '<br>' . wpforo_url( 'subscriptions', 'member' ),
						"description_original" => "For member profile, account and subscription pages use following URLs:",
						"docurl"               => "",
						"options_keys"         => [ 'redirect_url_after_login', 'redirect_url_after_register', 'redirect_url_after_confirm_sbscrb' ],
					],
					"redirect_url_after_login"          => [
						"type"                 => "text",
						"placeholder"          => "Redirect after login",
						"label"                => "",
						"label_original"       => "",
						"description"          => "",
						"description_original" => "",
					],
					"redirect_url_after_register"       => [
						"type"                 => "text",
						"placeholder"          => "Redirect after registration",
						"label"                => "",
						"label_original"       => "",
						"description"          => "",
						"description_original" => "",
					],
					"redirect_url_after_confirm_sbscrb" => [
						"type"                 => "text",
						"placeholder"          => "Redirect after subscription confirmation",
						"label"                => "",
						"label_original"       => "",
						"description"          => "",
						"description_original" => "",
					],
					"fb_api_config"                     => [
						"type"                 => "wrap",
						"label"                => esc_html__( 'Facebook API Configuration', 'wpforo' ),
						"label_original"       => "Facebook API Configuration",
						"description"          => __(
							                          'In order to get an App ID and Secret Key from Facebook, you’ll need to register a new application. Don’t worry – its very easy, and your application doesn\'t need to do anything. We only need the keys.',
							                          'wpforo'
						                          ) . sprintf(
							                          '&nbsp;<a href="https://wpforo.com/community/faq/how-to-get-facebook-app-id-and-secret-key/" target="_blank">%1$s &raquo;</a>',
							                          __( 'Please follow to this instruction', 'wpforo' )
						                          ),
						"description_original" => "In order to get an App ID and Secret Key from Facebook, you’ll need to register a new application. Don’t worry – its very easy, and your application doesn\'t need to do anything. We only need the keys.",
						"docurl"               => "",
						"options_keys"         => [ 'fb_api_id', 'fb_api_secret' ],
					],
					"fb_api_id"                         => [
						"type"                 => "text",
						"placeholder"          => esc_attr__( 'App ID', 'wpforo' ),
						"label"                => "",
						"label_original"       => "",
						"description"          => "",
						"description_original" => "",
					],
					"fb_api_secret"                     => [
						"type"                 => "text",
						"placeholder"          => esc_attr__( 'App Secret', 'wpforo' ),
						"label"                => "",
						"label_original"       => "",
						"description"          => "",
						"description_original" => "",
					],
					"fb_login"                          => [
						"type"                 => "radio",
						"label"                => esc_html__( "Facebook Login", "wpforo" ),
						"label_original"       => "Facebook Login",
						"description"          => esc_html__( "Adds Facebook Login button on Registration and Login pages.", "wpforo" ),
						"description_original" => "Adds Facebook Login button on Registration and Login pages.",
						"docurl"               => "",
					],
					"fb_lb_on_lp"                       => [
						"type"                 => "radio",
						"label"                => esc_html__( "Facebook Login button on User Login page", "wpforo" ),
						"label_original"       => "Facebook Login button on User Login page",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"fb_lb_on_rp"                       => [
						"type"                 => "radio",
						"label"                => esc_html__( "Facebook Login button on User Registration page", "wpforo" ),
						"label_original"       => "Facebook Login button on User Registration page",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"fb_redirect"                       => [
						"type"                 => "radio",
						"label"                => esc_html__( "Redirect to this page after success login", "wpforo" ),
						"label_original"       => "Redirect to this page after success login",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'profile', 'label' => 'Profile' ],
							[ 'value' => 'home', 'label' => 'Forums' ],
							[ 'value' => 'custom', 'label' => 'Custom' ],
						],
						"docurl"               => "",
					],
					"fb_redirect_url"                   => [
						"type"                 => "text",
						"placeholder"          => __( 'Custom URL, e.g.:', 'wpforo' ) . ' http://example.com/my-page/',
						"label"                => esc_html__( "Redirect to custom URL after success login", "wpforo" ),
						"label_original"       => "Redirect to custom URL after success login",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
				],
			],
			'email'         => [
				"title"                => esc_html__( "Email Settings", "wpforo" ),
				"title_original"       => "Email Settings",
				"icon"                 => '<svg viewBox="0 0 32 32" height="55px" xmlns="http://www.w3.org/2000/svg"><defs><style>.cls-33423{fill:none;stroke:#43a6df;stroke-linejoin:round;stroke-width:2px;}</style></defs><title/><g data-name="320-Email List" id="_320-Email_List"><polyline class="cls-33423" points="15 27 1 27 1 5 31 5 31 15"/><polyline class="cls-33423" points="1 5 16 16 31 5"/><circle class="cls-33423" cx="20" cy="20" r="1"/><line class="cls-33423" x1="24" x2="30" y1="20" y2="20"/><circle class="cls-33423" cx="20" cy="26" r="1"/><line class="cls-33423" x1="24" x2="30" y1="26" y2="26"/></g></svg>',
				"description"          => esc_html__(
					"Here, you can configure header information (FROM name and email) of forum specific emails. You can set administrators and moderators email addresses to get email notification when a new topic or reply is posted. Here you can also find user registration, password resetting and post reporting email templates.",
					"wpforo"
				),
				"description_original" => "Here, you can configure header information (FROM name and email) of forum specific emails. You can set administrators and moderators email addresses to get email notification when a new topic or reply is posted. Here you can also find user registration, password resetting and post reporting email templates.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/email-settings/",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/email.php' );
				},
				"options"              => [
					"from_name"                                          => [
						"type"                 => "text",
						"label"                => esc_html__( "FROM Name", "wpforo" ),
						"label_original"       => "FROM Name",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/email-settings/#from-name-and-email-address",
					],
					"from_email"                                         => [
						"type"                 => "text",
						"label"                => esc_html__( "FROM Email Address", "wpforo" ),
						"label_original"       => "FROM Email Address",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/email-settings/#from-name-and-email-address",
					],
					"admin_emails"                                       => [
						"type"                 => "text_,",
						"label"                => esc_html__( "Forum Admins Email Addresses", "wpforo" ),
						"label_original"       => "Forum Admins Email Addresses",
						"description"          => esc_html__( "Comma separated email addresses of forum administrators to get forum notifications. For example post report messages.", "wpforo" ),
						"description_original" => "Comma separated email addresses of forum administrators to get forum notifications. For example post report messages.",
						"docurl"               => "",
					],
					"new_topic_notify"                                   => [
						"type"                 => "radio",
						"label"                => esc_html__( "Notify Admins via Email on New Topic", "wpforo" ),
						"label_original"       => "Notify Admins via Email on New Topic",
						"description"          => esc_html__( "Send Notification emails to all email addresses (comma separated ) of forum administrators when a new Topic is created.", "wpforo" ),
						"description_original" => "Send Notification emails to all email addresses (comma separated ) of forum administrators when a new Topic is created.",
						"docurl"               => "",
					],
					"new_reply_notify"                                   => [
						"type"                 => "radio",
						"label"                => esc_html__( "Notify Admins via Email on New Post", "wpforo" ),
						"label_original"       => "Notify Admins via Email on New Post",
						"description"          => esc_html__( "Send Notification emails to all email addresses (comma separated ) of forum administrators when a new Reply is created.", "wpforo" ),
						"description_original" => "Send Notification emails to all email addresses (comma separated ) of forum administrators when a new Reply is created.",
						"docurl"               => "",
					],
					"disable_new_user_admin_notification"                => [
						"type"                 => "radio",
						"label"                => esc_html__( "Stop Sending Emails to Admins on New User Registration", "wpforo" ),
						"label_original"       => "Stop Sending Emails to Admins on New User Registration",
						"description"          => esc_html__(
							"If you have enabled this option, after registering, the admin will not receive an email notification for newly registered user.",
							"wpforo"
						),
						"description_original" => "If you have enabled this option, after registering, the admin will not receive an email notification for newly registered user.",
						"docurl"               => "",
					],
					"report_email_subject"                               => [
						"type"                 => "text",
						"label"                => esc_html__( "Report Message Subject", "wpforo" ),
						"label_original"       => "Report Message Subject",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"report_email_message"                               => [
						"type"                 => "richeditor",
						"media_buttons"        => true,
						"label"                => esc_html__( "Report Message Body", "wpforo" ),
						"label_original"       => "Report Message Body",
						"description"          => sprintf(
							'<ul class="wpf-email-shortcodes">
                               <li>[reporter] - %1$s</li>
                               <li>[message] - %2$s</li>
                               <li>[post_url] - %3$s</li>
                           </ul>',
							__( 'Reporter user display name', 'wpforo' ),
							__( 'Reporter user message', 'wpforo' ),
							__( 'Reported post URL', 'wpforo' )
						),
						"description_original" => "[reporter] - Reporter user display name [message] - Reporter user message [post_url] - Reported post URL",
						"docurl"               => "https://wpforo.com/community/faq/wpforo-email-shortcodes/",
					],
					"overwrite_new_user_notification_admin"              => [
						"type"                 => "radio",
						"label"                => esc_html__( "Overwrite WordPress New User Registration Email for Admins", "wpforo" ),
						"label_original"       => "Overwrite WordPress New User Registration Email for Admins",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"wp_new_user_notification_email_admin_subject"       => [
						"type"                 => "text",
						"label"                => esc_html__( "Message Subject", "wpforo" ),
						"label_original"       => "Message Subject",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"wp_new_user_notification_email_admin_message"       => [
						"type"                 => "richeditor",
						"media_buttons"        => true,
						"label"                => esc_html__( "Message Body", "wpforo" ),
						"label_original"       => "Message Body",
						"description"          => sprintf(
							'<ul class="wpf-email-shortcodes">
                                <li>[blogname] - %1$s</li>
                                <li>[user_login] - %2$s</li>
                                <li>[user_email] - %3$s</li>
                           </ul>',
							__( 'Website name', 'wpforo' ),
							__( 'Registered user login', 'wpforo' ),
							__( 'Registered user email', 'wpforo' )
						),
						"description_original" => "[blogname] - Website name [user_login] - Registered user login [user_email] - Registered user email",
						"docurl"               => "",
					],
					"after_user_delete_notification_email_admin_subject" => [
						"type"                 => "text",
						"label"                => esc_html__( "Message Subject", "wpforo" ),
						"label_original"       => "Message Subject",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"after_user_delete_notification_email_admin_message" => [
						"type"                 => "richeditor",
						"media_buttons"        => true,
						"label"                => esc_html__( "Message Body", "wpforo" ),
						"label_original"       => "Message Body",
						"description"          => sprintf(
							'<ul class="wpf-email-shortcodes">
                                <li>[user_login] - %1$s</li>
                                <li>[user_email] - %2$s</li>
                                <li>[user_userid] - %3$s</li>
                                <li>[datetime] - %4$s</li>
                           </ul>',
							__( 'Deleted user login', 'wpforo' ),
							__( 'Deleted user email', 'wpforo' ),
							__( 'Deleted user id', 'wpforo' ),
							__( 'Delete action date and time', 'wpforo' )
						),
						"description_original" => "[user_login] - Deleted user login [user_email] - Deleted user email [user_userid] - Deleted user id - Delete action date and time [datetime]",
						"docurl"               => "",
					],
					"overwrite_new_user_notification"                    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Overwrite WordPress New User Registration Email for Users", "wpforo" ),
						"label_original"       => "Overwrite WordPress New User Registration Email for Users",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"wp_new_user_notification_email_subject"             => [
						"type"                 => "text",
						"label"                => esc_html__( "Message Subject", "wpforo" ),
						"label_original"       => "Message Subject",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"wp_new_user_notification_email_message"             => [
						"type"                 => "richeditor",
						"media_buttons"        => true,
						"label"                => esc_html__( "Message Body", "wpforo" ),
						"label_original"       => "Message Body",
						"description"          => sprintf(
							'<ul class="wpf-email-shortcodes">
                               <li>[user_login] - %1$s</li>
                               <li>[set_password_url] - %2$s</li>
                           </ul>',
							__( 'Registered user login', 'wpforo' ),
							__( 'Link to open password reset form', 'wpforo' )
						),
						"description_original" => "[user_login] - Registered user login [set_password_url] - Link to open password reset form",
						"docurl"               => "https://wpforo.com/community/faq/wpforo-email-shortcodes/",
					],
					"overwrite_reset_password_email"                     => [
						"type"                 => "radio",
						"label"                => esc_html__( "Overwrite WordPress Reset Password Emails", "wpforo" ),
						"label_original"       => "Overwrite WordPress Reset Password Emails",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"reset_password_email_message"                       => [
						"type"                 => "richeditor",
						"media_buttons"        => true,
						"label"                => esc_html__( "Reset Password message body", "wpforo" ),
						"label_original"       => "Reset Password message body",
						"description"          => sprintf(
							'<ul class="wpf-email-shortcodes">
                               <li>[user_login] - %1$s</li>
                               <li>[reset_password_url] - %2$s</li>
                           </ul>',
							__( 'Registered user login', 'wpforo' ),
							__( 'Link to open password reset form', 'wpforo' )
						),
						"description_original" => "[user_login] - Registered user login [reset_password_url] - Link to open password reset form",
						"docurl"               => "https://wpforo.com/community/faq/wpforo-email-shortcodes/",
					],
					"after_user_approve_email_subject"                   => [
						"type"                 => "text",
						"label"                => esc_html__( "Message Subject", "wpforo" ),
						"label_original"       => "Message Subject",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"after_user_approve_email_message"                   => [
						"type"                 => "richeditor",
						"media_buttons"        => true,
						"label"                => esc_html__( "Message Body", "wpforo" ),
						"label_original"       => "Message Body",
						"description"          => sprintf(
							'<ul class="wpf-email-shortcodes">
                               <li>[blogname] - %1$s</li>
                               <li>[user_login] - %2$s</li>
                               <li>[login_link] - %3$s</li>
                               <li>[login_url]  - %4$s</li>
                           </ul>',
							__( 'WordPress Blog Name', 'wpforo' ),
							__( 'Approved User Login', 'wpforo' ),
							__( 'Approved User Login Link', 'wpforo' ),
							__( 'Approved User Login URL', 'wpforo' )
						),
						"description_original" => "[user_login] - Approved User Login [login_link] - Approved User Login Link [login_url] - Approved User Login URL ",
						"docurl"               => "",
					],
				],
			],
			'subscriptions' => [
				"title"                => esc_html__( "Subscriptions &amp; Mentioning", "wpforo" ),
				"title_original"       => "Subscriptions & Mentioning",
				"icon"                 => '<svg height="54px" data-name="Layer 1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 117.88 122.88"><path d="M102.31,30.86,78,5.74V5.49H12.44V46h0L46.66,76.6l.15.14L58.36,87.06,105,46.36v-10l-2.64-5.49ZM64.07,28.44l.09-.41.56-2.47h6.83L67.62,44.1c-.07.33-.14.63-.19.9h0a9.34,9.34,0,0,0-.18,1.42,1.43,1.43,0,0,0,.29.94,1,1,0,0,0,.72.33c.93,0,2.14-.61,3.61-1.82a16.05,16.05,0,0,0,3.85-5,15.19,15.19,0,0,0,.47-12.71,13.86,13.86,0,0,0-3.4-4.76,16,16,0,0,0-5.5-3.15,22.48,22.48,0,0,0-7.41-1.14,25.87,25.87,0,0,0-7.73,1.11A19.49,19.49,0,0,0,46,23.41a17.46,17.46,0,0,0-4.45,5.08,22.14,22.14,0,0,0-2.4,5.84,25.72,25.72,0,0,0-.83,6.54A18.36,18.36,0,0,0,40.07,49a16.3,16.3,0,0,0,5.14,6.18,21,21,0,0,0,8.16,3.55,34.53,34.53,0,0,0,10.8.42A25.52,25.52,0,0,0,72.27,57a15.69,15.69,0,0,0,5.63-4.5h5.51a18.14,18.14,0,0,1-3.5,5,20.31,20.31,0,0,1-5.06,3.66,26.25,26.25,0,0,1-6.49,2.26,36.89,36.89,0,0,1-7.84.77,36.4,36.4,0,0,1-11.44-1.67,23.41,23.41,0,0,1-8.58-4.81,20.32,20.32,0,0,1-5.27-7.5,25.24,25.24,0,0,1-1.77-9.62A28.31,28.31,0,0,1,35.3,30.22,23.54,23.54,0,0,1,49,16.39a29.27,29.27,0,0,1,11-2,27,27,0,0,1,9,1.44A19.93,19.93,0,0,1,76,20a18.12,18.12,0,0,1,4.47,6.38A21,21,0,0,1,82,34.46a19.13,19.13,0,0,1-1.24,6.78,19.37,19.37,0,0,1-3.56,6,17.25,17.25,0,0,1-5.44,4.19A15.06,15.06,0,0,1,65,53a6.17,6.17,0,0,1-3.25-.72A3.48,3.48,0,0,1,60.22,50,12,12,0,0,1,59,51a10.61,10.61,0,0,1-1.06.68,9.49,9.49,0,0,1-2.89,1.07,9.65,9.65,0,0,1-5.67-.55A9.38,9.38,0,0,1,45,48.44a12,12,0,0,1-1.69-6.56,19.55,19.55,0,0,1,1.74-8,16.59,16.59,0,0,1,5-6.44,11.56,11.56,0,0,1,7.29-2.51A7.46,7.46,0,0,1,64,28.25l.12.19ZM49.93,41.83a7.62,7.62,0,0,0,1.29,4.76,4.1,4.1,0,0,0,3.38,1.66,5.75,5.75,0,0,0,4.28-2,10,10,0,0,0,1-1.23,10.61,10.61,0,0,0,.6-.93,13.67,13.67,0,0,0,1.2-2.74,18.71,18.71,0,0,0,1-5.66v-.07a10.84,10.84,0,0,0-.06-1.16,7.48,7.48,0,0,0-.26-1.3,5.92,5.92,0,0,0-1-1.93A4.18,4.18,0,0,0,59.84,30a4.76,4.76,0,0,0-2-.42,5.74,5.74,0,0,0-4.1,1.78,11.34,11.34,0,0,0-2.78,4.54h0l0,0a17.92,17.92,0,0,0-1,5.86Zm60.51-.26a3.59,3.59,0,0,1,2.43-.93,4,4,0,0,1,2.06.6,5.09,5.09,0,0,1,1.26,1.07,7.06,7.06,0,0,1,1.69,4.26v70.64a5.71,5.71,0,0,1-1.66,4h0a5.67,5.67,0,0,1-4,1.66H5.67a5.71,5.71,0,0,1-4-1.66h0a5.62,5.62,0,0,1-1.66-4V46.57a7.1,7.1,0,0,1,1.73-4.32,5.5,5.5,0,0,1,1.26-1,4,4,0,0,1,2-.58,3.59,3.59,0,0,1,2,.57V2.74A2.74,2.74,0,0,1,9.7,0H78.9A2.71,2.71,0,0,1,81,1l28.65,29.59a2.71,2.71,0,0,1,.78,1.9h0v.79c0,.11,0,.22,0,.34s0,.22,0,.33v7.63ZM56.52,92.74,44.9,82.36,5.49,116.9v.31a.17.17,0,0,0,0,.13h0a.22.22,0,0,0,.13,0H112.21a.22.22,0,0,0,.13,0h0a.17.17,0,0,0,0-.13V117L72.45,82,60.15,92.76h0a2.73,2.73,0,0,1-3.62,0Zm-15.73-14L5.49,47.15v62.48l35.3-30.94Zm71.6,31.06V47.12L76.6,78.39l35.79,31.36Z"/></svg>',
				"description"          => esc_html__(
					"wpForo has almost all kind of way to keep forum users updated about the latest posted content. Users can subscribe to forums and topics, they can be mentioned in posts, they can follow to other users. On all these cases, users get email and in-forum notification. And all these features can be managed in this admin section.",
					"wpforo"
				),
				"description_original" => "wpForo has almost all kind of way to keep forum users updated about the latest posted content. Users can subscribe to forums and topics, they can be mentioned in posts, they can follow to other users. On all these cases, users get email and in-forum notification. And all these features can be managed in this admin section.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/subscriptions-mentioning/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/subscriptions.php' );
				},
				"options"              => [
					"subscribe_confirmation"               => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Subscription Confirmation", "wpforo" ),
						"label_original"       => "Enable Subscription Confirmation",
						"description"          => esc_html__( "Forum and Topic subscription with double opt-in/confirmation system.", "wpforo" ),
						"description_original" => "Forum and Topic subscription with double opt-in/confirmation system.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/subscriptions-mentioning/#enable-subscription-confirmation",
					],
					"subscribe_checkbox_on_post_editor"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Topic subscription option on post editor", "wpforo" ),
						"label_original"       => "Topic subscription option on post editor",
						"description"          => esc_html__( "This option adds topic subscription checkbox next to new topic and post submit button.", "wpforo" ),
						"description_original" => "This option adds topic subscription checkbox next to new topic and post submit button.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/subscriptions-mentioning/#topic-subscription-option-on-post-editor",
					],
					"subscribe_checkbox_default_status"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Topic subscription option on post editor - checked/enabled", "wpforo" ),
						"label_original"       => "Topic subscription option on post editor - checked/enabled",
						"description"          => esc_html__( "Enable this option if you want the topic subscription checkbox to be checked by default.", "wpforo" ),
						"description_original" => "Enable this option if you want the topic subscription checkbox to be checked by default.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/subscriptions-mentioning/#topic-subscription-option-on-post-editor",
					],
					"user_mention_notify"                  => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable User Mentioning", "wpforo" ),
						"label_original"       => "Enable User Mentioning",
						"description"          => esc_html__( "If this option is enabled, users will receive email notification when someone uses @nickname of the user in posts.", "wpforo" ),
						"description_original" => "If this option is enabled, users will receive email notification when someone uses @nickname of the user in posts.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/subscriptions-mentioning/#enable-user-mentioning",
					],
					"user_following_notify"                => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable User Following", "wpforo" ),
						"label_original"       => "Enable User Following",
						"description"          => esc_html__(
							"If this option is enabled, the [Follow] button will be displayed on user's profile pages, allowing users to subscribe to certain users topics and posts. The follower user will receive email notification when the followed user posts a new topic or post.",
							"wpforo"
						),
						"description_original" => "If this option is enabled, the [Follow] button will be displayed on user's profile pages, allowing users to subscribe to certain users topics and posts. The follower user will receive email notification when the followed user posts a new topic or post.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/subscriptions-mentioning/#enable-user-following",
					],
					"confirmation_email_subject"           => [
						"type"                 => "text",
						"label"                => esc_html__( "Subscribe confirmation email subject", "wpforo" ),
						"label_original"       => "Subscribe confirmation email subject",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"confirmation_email_message"           => [
						"type"                 => "richeditor",
						"media_buttons"        => true,
						"label"                => esc_html__( "Subscribe confirmation email message", "wpforo" ),
						"label_original"       => "Subscribe confirmation email message",
						"description"          => sprintf(
							'<ul class="wpf-email-shortcodes">
                               <li>[entry_title] - %1$s</li>
                               <li>[user_display_name] - %2$s</li>
                               <li>[confirm_link] - %3$s</li>
                           </ul>',
							__( 'Subscribed forum or topic title', 'wpforo' ),
							__( 'Subscriber display name', 'wpforo' ),
							__( 'Link to confirm subscription', 'wpforo' )
						),
						"description_original" => "[entry_title] - Subscribed forum or topic title [user_display_name] - Subscriber display name [confirm_link] - Link to confirm subscription",
						"docurl"               => "https://wpforo.com/community/faq/wpforo-email-shortcodes/",
					],
					"new_topic_notification_email_subject" => [
						"type"                 => "text",
						"label"                => esc_html__( "New topic notification email subject", "wpforo" ),
						"label_original"       => "New topic notification email subject",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"new_topic_notification_email_message" => [
						"type"                 => "richeditor",
						"media_buttons"        => true,
						"label"                => esc_html__( "New topic notification email message", "wpforo" ),
						"label_original"       => "New topic notification email message",
						"description"          => sprintf(
							'<ul class="wpf-email-shortcodes">
                               <li>[user_display_name] - %1$s</li>
                               <li>[owner_display_name] - %2$s</li>
                               <li>[forum_link] - %3$s</li>
                               <li>[topic_link] - %4$s</li>
                               <li>[topic_title] - %5$s</li>
                               <li>[topic_body] - %6$s</li>
                               <li>[unsubscribe_link] - %7$s</li>
                           </ul>',
							__( 'Subscriber display name', 'wpforo' ),
							__( 'New topic author display name', 'wpforo' ),
							__( 'Forum title / link', 'wpforo' ),
							__( 'Topic title / link', 'wpforo' ),
							__( 'New topic title', 'wpforo' ),
							__( 'New topic excerpt', 'wpforo' ),
							__( 'Link to unsubscribe', 'wpforo' )
						),
						"description_original" => "[user_display_name] - Subscriber display name [owner_display_name] - New topic author display name [forum_link] - Forum title / link [topic_link] - Topic title / link [topic_title] - New topic title [topic_body] - New topic excerpt [unsubscribe_link] - Link to unsubscribe",
						"docurl"               => "https://wpforo.com/community/faq/wpforo-email-shortcodes/",
					],
					"new_post_notification_email_subject"  => [
						"type"                 => "text",
						"label"                => esc_html__( "New reply notification email subject", "wpforo" ),
						"label_original"       => "New reply notification email subject",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"new_post_notification_email_message"  => [
						"type"                 => "richeditor",
						"media_buttons"        => true,
						"label"                => esc_html__( "New reply notification email message", "wpforo" ),
						"label_original"       => "New reply notification email message",
						"description"          => sprintf(
							'<ul class="wpf-email-shortcodes">
                               <li>[user_display_name] - %1$s</li>
                               <li>[owner_display_name] - %2$s</li>
                               <li>[topic_link] - %3$s</li>
                               <li>[post_title] - %4$s</li>
                               <li>[post_body] - %5$s</li>
                               <li>[unsubscribe_link] - %6$s</li>
                           </ul>',
							__( 'Subscriber display name', 'wpforo' ),
							__( 'New reply author display name', 'wpforo' ),
							__( 'Topic title / link', 'wpforo' ),
							__( 'New reply title', 'wpforo' ),
							__( 'New reply excerpt', 'wpforo' ),
							__( 'Link to unsubscribe', 'wpforo' )
						),
						"description_original" => "[user_display_name] - Subscriber display name [owner_display_name] - New reply author display name [topic_link] - Topic title / link [post_title] - New reply title [post_body] - New reply excerpt [unsubscribe_link] - Link to unsubscribe",
						"docurl"               => "https://wpforo.com/community/faq/wpforo-email-shortcodes/",
					],
					"user_mention_email_subject"           => [
						"type"                 => "text",
						"label"                => esc_html__( "User mention message subject", "wpforo" ),
						"label_original"       => "User mention message subject",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"user_mention_email_message"           => [
						"type"                 => "richeditor",
						"media_buttons"        => true,
						"label"                => esc_html__( "User Mention message body", "wpforo" ),
						"label_original"       => "User Mention message body",
						"description"          => sprintf(
							'<ul class="wpf-email-shortcodes">
                               <li>[user_display_name] - %1$s</li>
                               <li>[owner_display_name] - %2$s</li>
                               <li>[topic_title] - %3$s</li>
                               <li>[post_url] - %4$s</li>
                               <li>[post_body] - %5$s</li>
                           </ul>',
							__( 'Mentioned user display name', 'wpforo' ),
							__( 'Post author display name', 'wpforo' ),
							__( 'Topic title', 'wpforo' ),
							__( 'Link to the post', 'wpforo' ),
							__( 'Mentioned post excerpt', 'wpforo' )
						),
						"description_original" => "[user_display_name] - Mentioned user display name [owner_display_name] - Post author display name [topic_title] - Topic title [post_url] - Link to the post [post_body] - Mentioned post excerpt",
						"docurl"               => "https://wpforo.com/community/faq/wpforo-email-shortcodes/",
					],
					"user_following_email_subject"         => [
						"type"                 => "text",
						"label"                => esc_html__( "User following notification email subject", "wpforo" ),
						"label_original"       => "User following notification email subject",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"user_following_email_message"         => [
						"type"                 => "richeditor",
						"media_buttons"        => true,
						"label"                => esc_html__( "User following notification email message", "wpforo" ),
						"label_original"       => "User following notification email message",
						"description"          => sprintf(
							'<ul class="wpf-email-shortcodes">
                                <li>[user_display_name] - %1$s</li>
                                <li>[owner_display_name] - %2$s</li>
                                <li>[post_title] - %3$s</li>
                                <li>[post_body] - %4$s</li>
                                <li>[unsubscribe_link] - %5$s</li>
                            </ul>',
							__( 'Subscriber display name', 'wpforo' ),
							__( 'New post author display name', 'wpforo' ),
							__( 'New post title', 'wpforo' ),
							__( 'New post excerpt', 'wpforo' ),
							__( 'Link to unsubscribe', 'wpforo' )
						),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/community/faq/wpforo-email-shortcodes/",
					],
				],
			],
			'notifications' => [
				"title"                => esc_html__( "Forum Notifications", "wpforo" ),
				"title_original"       => "Forum Notifications",
				"icon"                 => '<svg version="1.1" height="50px" xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 116.11 122.88" style="enable-background:new 0 0 116.11 122.88" xml:space="preserve"><g><path d="M74.82,109.04c-0.37,1.94-1.02,3.72-1.96,5.35c-0.97,1.67-2.24,3.18-3.8,4.5c-1.57,1.32-3.27,2.32-5.12,2.99 c-1.85,0.67-3.81,1-5.88,1c-2.08,0-4.04-0.34-5.88-1c-1.84-0.67-3.55-1.66-5.11-2.99c-1.57-1.33-2.83-2.83-3.8-4.5 c-0.97-1.67-1.63-3.51-1.99-5.53c-0.18-0.98,0.48-1.92,1.46-2.1c0.03,0,0.32-0.03,0.32-0.03h30.02c1,0,1.82,0.81,1.82,1.82 C74.89,108.72,74.86,108.88,74.82,109.04L74.82,109.04L74.82,109.04z M20.21,0.25c1.83-0.73,3.9,0.17,4.63,2 c0.73,1.83-0.17,3.9-2,4.63c-3.96,1.58-7.28,3.77-9.93,6.61c-2.64,2.84-4.63,6.36-5.93,10.59c-0.58,1.88-2.58,2.94-4.46,2.36 c-1.88-0.58-2.94-2.58-2.36-4.46c1.63-5.3,4.15-9.74,7.52-13.36C11.05,5.01,15.24,2.23,20.21,0.25L20.21,0.25z M93.27,6.88 c-1.83-0.73-2.73-2.8-2-4.63c0.73-1.83,2.8-2.73,4.63-2c4.97,1.98,9.16,4.76,12.53,8.38c3.37,3.63,5.9,8.07,7.52,13.36 c0.58,1.88-0.48,3.88-2.36,4.46c-1.88,0.58-3.88-0.48-4.46-2.36c-1.3-4.24-3.29-7.76-5.93-10.59 C100.55,10.65,97.23,8.46,93.27,6.88L93.27,6.88z M67.62,10.54c1.47,0.38,2.9,0.85,4.29,1.4c2.04,0.81,4,1.78,5.88,2.91 c0.07,0.05,0.15,0.09,0.22,0.14c1.8,1.11,3.48,2.33,5.02,3.65c1.62,1.39,3.12,2.92,4.52,4.6l0.01,0.01h0 c1.37,1.65,2.59,3.42,3.67,5.29c1.08,1.88,2.01,3.84,2.78,5.86l0,0c0.79,2.09,1.38,4.22,1.78,6.41c0.39,2.2,0.59,4.45,0.59,6.76 c0,4.56,0,7.03,0,7.33c0.01,2.34,0.02,4.63,0.04,6.86v0.02l0,0c0.01,2.02,0.14,4.05,0.39,6.08c0.25,2.01,0.61,3.95,1.08,5.82l0,0 c0.47,1.84,1.11,3.62,1.9,5.32c0.82,1.75,1.82,3.47,2.99,5.14l0.01,0c1.16,1.64,2.61,3.27,4.35,4.87c1.8,1.65,3.88,3.28,6.26,4.86 c1.36,0.91,1.73,2.76,0.81,4.12c-0.57,0.85-1.51,1.32-2.47,1.32v0.01l-26.85,0H58.06H31.21H4.37c-1.65,0-2.98-1.33-2.98-2.98 c0-1.08,0.58-2.03,1.44-2.55c2.41-1.63,4.48-3.25,6.21-4.85c1.72-1.59,3.16-3.22,4.32-4.9c0.03-0.05,0.07-0.1,0.11-0.14 c1.12-1.64,2.08-3.31,2.87-5.01c0.81-1.73,1.46-3.51,1.94-5.34c0.01-0.04,0.02-0.08,0.03-0.11c0.46-1.78,0.81-3.66,1.05-5.63 c0.24-1.98,0.37-4.03,0.37-6.14v-14.1c0-2.27,0.2-4.52,0.61-6.77c0.41-2.24,1-4.39,1.79-6.44c0.78-2.05,1.72-4.02,2.81-5.9 c1.08-1.87,2.32-3.64,3.71-5.32l0.02-0.02l0,0c1.38-1.65,2.9-3.19,4.55-4.6c1.63-1.39,3.39-2.66,5.28-3.79 c1.91-1.14,3.89-2.1,5.93-2.88c1.42-0.54,2.89-0.99,4.39-1.36c0.51-1.79,1.39-3.24,2.64-4.35c1.72-1.53,3.98-2.29,6.79-2.26 c2.78,0.02,5.03,0.79,6.73,2.32C66.22,7.32,67.11,8.76,67.62,10.54L67.62,10.54L67.62,10.54z M69.75,17.47 c-1.65-0.65-3.33-1.16-5.04-1.53c-1.32-0.17-2.4-1.21-2.57-2.59c-0.16-1.3-0.53-2.21-1.12-2.73c-0.59-0.52-1.53-0.79-2.82-0.8 c-1.29-0.01-2.22,0.24-2.79,0.75c-0.58,0.52-0.95,1.44-1.1,2.76h0c-0.14,1.26-1.09,2.34-2.41,2.58c-1.85,0.35-3.64,0.85-5.37,1.51 c-1.73,0.65-3.38,1.46-4.98,2.41c-1.59,0.95-3.08,2.02-4.46,3.21c-1.38,1.18-2.67,2.48-3.85,3.9l0,0c-1.16,1.4-2.2,2.91-3.13,4.51 c-0.91,1.58-1.71,3.26-2.39,5.04c-0.68,1.77-1.18,3.57-1.51,5.37c-0.33,1.81-0.49,3.72-0.49,5.72v14.1c0,2.34-0.14,4.62-0.41,6.86 c-0.27,2.15-0.67,4.29-1.22,6.4c-0.01,0.05-0.02,0.09-0.03,0.14c-0.57,2.15-1.34,4.26-2.31,6.34c-0.94,2.01-2.06,3.96-3.35,5.85 c-0.04,0.06-0.08,0.12-0.12,0.18c-1.36,1.96-3.09,3.91-5.18,5.85l-0.08,0.07h18.22h26.85H84.9h18.19c-2.04-1.88-3.76-3.82-5.16-5.8 l0,0l0-0.01c-1.37-1.96-2.54-3.97-3.51-6.03c-0.99-2.1-1.75-4.23-2.3-6.37l0,0l0-0.01c-0.54-2.13-0.95-4.32-1.22-6.56 c-0.26-2.14-0.4-4.4-0.41-6.77v-0.01c-0.02-2.21-0.03-4.51-0.04-6.91c-0.02-4.44-0.02-6.86-0.02-7.33c0-1.96-0.16-3.87-0.5-5.72 c-0.33-1.84-0.82-3.62-1.47-5.34l0,0l0-0.01c-0.67-1.77-1.46-3.44-2.36-5.01c-0.9-1.57-1.94-3.06-3.11-4.48l0,0 c-1.16-1.39-2.43-2.68-3.81-3.87c-1.34-1.15-2.76-2.18-4.25-3.11c-0.07-0.03-0.13-0.07-0.2-0.11 C73.11,18.97,71.45,18.15,69.75,17.47L69.75,17.47L69.75,17.47z"/></g></svg>',
				"description"          => esc_html__(
					"As an addition to email notification functions, wpForo has in-forum notification system. You can see the notification bell on forum menu bar or in the sidebar with 'My Profile and Notifications' widget.",
					"wpforo"
				),
				"description_original" => "As an addition to email notification functions, wpForo has in-forum notification system. You can see the notification bell on forum menu bar or in the sidebar with 'My Profile and Notifications' widget.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/forum-notifications/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/notifications.php' );
				},
				"options"              => [
					"notifications"      => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable User Notification", "wpforo" ),
						"label_original"       => "Enable User Notification",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"notifications_live" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable User Notification Live Update", "wpforo" ),
						"label_original"       => "Enable User Notification Live Update",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/forum-notifications/#enable-user-notification-live-update",
					],
					"notifications_bar"  => [
						"type"                 => "radio",
						"label"                => esc_html__( "Display User Notification Bell on Menu Bar", "wpforo" ),
						"label_original"       => "Display User Notification Bell on Menu Bar",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/forum-notifications/#display-user-notification-bell-on-menu-bar",
					],
				],
			],
			'logging'       => [
				"title"                => esc_html__( "Action Logging &amp; Views", "wpforo" ),
				"title_original"       => "Action Logging & Views",
				"icon"                 => '<svg height="55px" data-name="Layer 1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 109.21 122.88"><title>mouse-click</title><path d="M86,122.31a5.57,5.57,0,0,1-.9.35,5.09,5.09,0,0,1-1,.18,5.46,5.46,0,0,1-1,0,6.77,6.77,0,0,1-1-.15,6,6,0,0,1-1-.36l0,0a5.51,5.51,0,0,1-.92-.53l0,0a6.41,6.41,0,0,1-.78-.69,5.19,5.19,0,0,1-.65-.87l-9.08-14.88-7.69,9a15.49,15.49,0,0,1-1.1,1.18c-.39.37-.78.71-1.18,1l-.08.06a12.19,12.19,0,0,1-1.2.82,9.66,9.66,0,0,1-1.24.63,6.91,6.91,0,0,1-1,.37,6.21,6.21,0,0,1-1,.22,7.55,7.55,0,0,1-1.06.07,7.19,7.19,0,0,1-1-.11,6.14,6.14,0,0,1-1.18-.35,5.42,5.42,0,0,1-1.06-.57,6.22,6.22,0,0,1-.92-.78l0,0a7.31,7.31,0,0,1-.75-1l-.11-.2-.09-.21L47.72,112l0-.17L40.91,43.26a4.52,4.52,0,0,1,0-1.33,4.3,4.3,0,0,1,.43-1.25,4.31,4.31,0,0,1,1.39-1.55l0,0a3.82,3.82,0,0,1,.9-.46,4.25,4.25,0,0,1,1-.24h0a4.31,4.31,0,0,1,1.29.05,4.67,4.67,0,0,1,1.25.44l.3.16c13.51,8.84,26.1,17.06,38.64,25.25l19,12.39a11.72,11.72,0,0,1,1,.72l0,0a8.78,8.78,0,0,1,.82.73l.06.07a7.41,7.41,0,0,1,.71.82,5.91,5.91,0,0,1,.57.87,6.42,6.42,0,0,1,.51,1.14,5.6,5.6,0,0,1,.26,1.17,5.44,5.44,0,0,1,0,1.21h0a6.59,6.59,0,0,1-.23,1.19,6.54,6.54,0,0,1-.94,1.88,6.41,6.41,0,0,1-.67.83,7.45,7.45,0,0,1-.82.76,10.42,10.42,0,0,1-1.16.83,12.92,12.92,0,0,1-1.34.7c-.47.21-1,.41-1.46.58a14.27,14.27,0,0,1-1.55.43h0c-2.77.54-5.53,1.21-8.27,1.87l-3.25.77,9,14.94a5.84,5.84,0,0,1,.46,1,5.59,5.59,0,0,1,.15,3.21l0,.1a5.53,5.53,0,0,1-.33.94,6.43,6.43,0,0,1-.51.89,5.62,5.62,0,0,1-.68.81,6,6,0,0,1-.82.67l-2,1.29A83,83,0,0,1,86,122.31ZM37.63,19.46a4,4,0,0,1-6.92,4l-8-14a4,4,0,0,1,6.91-4l8.06,14Zm-15,46.77a4,4,0,0,1,4,6.91l-14,8.06a4,4,0,0,1-4-6.91l14-8.06ZM20.56,39.84a4,4,0,0,1-2.07,7.72L3,43.36A4,4,0,0,1,5,35.64l15.53,4.2ZM82,41.17a4,4,0,0,1-4-6.91L92,26.2a4,4,0,0,1,4,6.91L82,41.17ZM63.46,20.57a4,4,0,1,1-7.71-2.06L59.87,3A4,4,0,0,1,67.59,5L63.46,20.57Zm20.17,96.36,9.67-5.86c-3.38-5.62-8.85-13.55-11.51-19.17a2.17,2.17,0,0,1-.12-.36,2.4,2.4,0,0,1,1.81-2.87c5.38-1.23,10.88-2.39,16.22-3.73a10.28,10.28,0,0,0,1.8-.58,6.11,6.11,0,0,0,1.3-.77,3.38,3.38,0,0,0,.38-.38.9.9,0,0,0,.14-.24l-.06-.18a2.15,2.15,0,0,0-.44-.53,5.75,5.75,0,0,0-.83-.63L47.06,45.75c2.11,21.36,5.2,44.1,6.45,65.31a6.28,6.28,0,0,0,.18,1,2.89,2.89,0,0,0,.26.62l.13.14a1,1,0,0,0,.29,0,2.76,2.76,0,0,0,.51-.17,5.71,5.71,0,0,0,1.28-.79,11.22,11.22,0,0,0,1.35-1.33c1.93-2.27,9.6-12.14,11.4-13.18a2.4,2.4,0,0,1,3.28.82l11.44,18.75Z"/></svg>',
				"description"          => esc_html__(
					"wpForo logging system allows tracking and providing live information about forum viewers. It detects read/unread status of forums and posts for guests and logged-in users individually. Here you can manage all functions and features of the forum logging system.",
					"wpforo"
				),
				"description_original" => "wpForo logging system allows tracking and providing live information about forum viewers. It detects read/unread status of forums and posts for guests and logged-in users individually. Here you can manage all functions and features of the forum logging system.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/action-logging-views/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/logging.php' );
				},
				"options"              => [
					"view_logging"                  => [
						"type"                 => "radio",
						"label"                => esc_html__( "Log Viewed Forums and Topics", "wpforo" ),
						"label_original"       => "Log Viewed Forums and Topics",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/action-logging-views/#log-viewed-forums-and-topics",
					],
					"track_logging"                 => [
						"type"                 => "radio",
						"label"                => esc_html__( "Track Forum and Topic Current Viewers", "wpforo" ),
						"label_original"       => "Track Forum and Topic Current Viewers",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/action-logging-views/#track-forum-and-topic-current-viewers",
					],
					"goto_unread"                   => [
						"type"                 => "radio",
						"label"                => esc_html__( "Topic Links - Jump to First Unread Post", "wpforo" ),
						"label_original"       => "Topic Links - Jump to First Unread Post",
						"description"          => esc_html__(
							"If this option is enabled, all topic links on forums page and on topics page will refer to the first unread post. This behavior is only enabled for logged-in users.",
							"wpforo"
						),
						"description_original" => "If this option is enabled, all topic links on forums page and on topics page will refer to the first unread post. This behavior is only enabled for logged-in users.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/action-logging-views/#topic-links-jump-to-first-unread-post",
					],
					"goto_unread_button"            => [
						"type"                 => "radio",
						"label"                => esc_html__( "Jump to First Unread Post with [new] Button in Topic Link", "wpforo" ),
						"label_original"       => "Jump to First Unread Post with [new] Button in Topic Link",
						"description"          => esc_html__( "Adds [new] button at the and of topic links, which jumps to the first unread post.", "wpforo" ),
						"description_original" => "Adds [new] button at the and of topic links, which jumps to the first unread post.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/action-logging-views/#jump-to-first-unread-post-with-new-button-in-topic-link",
					],
					"display_forum_current_viewers" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Display Forum Current Viewers", "wpforo" ),
						"label_original"       => "Display Forum Current Viewers",
						"description"          => esc_html__( "Displays information about forum current viewers (x viewing) next to forum title.", "wpforo" ),
						"description_original" => "Displays information about forum current viewers (x viewing) next to forum title.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/action-logging-views/#display-forum-and-topic-current-viewers",
					],
					"display_topic_current_viewers" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Display Topic Current Viewers", "wpforo" ),
						"label_original"       => "Display Topic Current Viewers",
						"description"          => esc_html__( "Displays information about topic current viewers (x viewing) next to topic title", "wpforo" ),
						"description_original" => "Displays information about topic current viewers (x viewing) next to topic title",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/action-logging-views/#display-forum-and-topic-current-viewers",
					],
					"display_recent_viewers"        => [
						"type"                 => "radio",
						"label"                => esc_html__( "Display Recent Viewers in the Topic Footer", "wpforo" ),
						"label_original"       => "Display Recent Viewers in the Topic Footer",
						"description"          => esc_html__( "Displays information about topic recent viewers (users visited within last one hour)", "wpforo" ),
						"description_original" => "Displays information about topic recent viewers (users visited within last one hour)",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/action-logging-views/#display-recent-viewers-in-the-topic-footer",
					],
					"display_admin_viewers"         => [
						"type"                 => "radio",
						"label"                => esc_html__( "Display Administrators in the Topic Recent Viewers Section", "wpforo" ),
						"label_original"       => "Display Administrators in the Topic Recent Viewers Section",
						"description"          => esc_html__( "By disabling this option you can exclude forum administrators from the topic viewers section.", "wpforo" ),
						"description_original" => "By disabling this option you can exclude forum administrators from the topic viewers section.",
						"docurl"               => "",
					],
				],
			],
			'seo'           => [
				"title"                => esc_html__( "wpForo SEO", "wpforo" ),
				"title_original"       => "wpForo SEO",
				"icon"                 => '<svg height="55px" data-name="Layer 1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 111.56 122.88"><defs><style>.cls-5514{fill-rule:evenodd;}</style></defs><title>testing</title><path class="cls-5514" d="M79.86,65.67a25,25,0,0,1,20.89,38.62l10.81,11.78-7.46,6.81L93.68,111.42A25,25,0,1,1,79.86,65.67Zm-42.65.26a2.74,2.74,0,0,1-2.6-2.84,2.71,2.71,0,0,1,2.6-2.84h15.4a2.76,2.76,0,0,1,2.6,2.84,2.71,2.71,0,0,1-2.6,2.84ZM22.44,57.22a5.67,5.67,0,1,1-5.67,5.67,5.67,5.67,0,0,1,5.67-5.67Zm2-18.58a2,2,0,0,1,2.85,0,2.07,2.07,0,0,1,0,2.89l-2,2,2,2a2,2,0,0,1,0,2.87,2,2,0,0,1-2.84,0l-2-2-2,2a2,2,0,0,1-2.86,0,2.07,2.07,0,0,1,0-2.89l2-2-2-2.05a2,2,0,0,1,2.87-2.86l2,2,2-2ZM16.85,21.52a2.29,2.29,0,0,1,3.16.63l1.13,1.36,4-5.05a2.27,2.27,0,1,1,3.51,2.88l-5.86,7.34a2.48,2.48,0,0,1-.55.52,2.28,2.28,0,0,1-3.16-.63l-2.84-3.89a2.28,2.28,0,0,1,.63-3.16Zm66.51-4.25h9.32a6.69,6.69,0,0,1,6.66,6.65v30.9c-.2,2.09-5.31,2.11-5.75,0V23.92a.93.93,0,0,0-.27-.67.91.91,0,0,0-.67-.27H83.32V54.82c-.49,1.89-4.75,2.18-5.71,0V6.66A1,1,0,0,0,77.34,6a.93.93,0,0,0-.67-.27h-70A.93.93,0,0,0,6,6a1,1,0,0,0-.27.68V85.79a1,1,0,0,0,.27.68.93.93,0,0,0,.67.27H44.74c2.88.29,3,5.27,0,5.71H21.66v10.61a.92.92,0,0,0,.94.94H44.74c2.09.24,2.76,5,0,5.71H22.64a6.54,6.54,0,0,1-4.7-2,6.63,6.63,0,0,1-2-4.7V92.45H6.66A6.69,6.69,0,0,1,0,85.79V6.66A6.54,6.54,0,0,1,2,2a6.61,6.61,0,0,1,4.7-2h70a6.55,6.55,0,0,1,4.7,2,6.65,6.65,0,0,1,2,4.7V17.27ZM37.18,26.44a2.75,2.75,0,0,1-2.6-2.84,2.71,2.71,0,0,1,2.6-2.84H63.86a2.74,2.74,0,0,1,2.6,2.84,2.71,2.71,0,0,1-2.6,2.84Zm0,19.74a2.74,2.74,0,0,1-2.6-2.83,2.71,2.71,0,0,1,2.6-2.84H63.86a2.74,2.74,0,0,1,2.6,2.84,2.7,2.7,0,0,1-2.6,2.83ZM70.45,93a3.46,3.46,0,0,1-.34-.44,3.4,3.4,0,0,1-.26-.5,3.18,3.18,0,0,1,4.57-4,2.93,2.93,0,0,1,.49.38h0c.87.83,1.15,1,2.11,1.87l.84.74,6.79-7.29c2.87-3,7.45,1.37,4.58,4.4l-8.47,9.06-.43.45a3.19,3.19,0,0,1-4.43.19l0,0c-.22-.19-.44-.4-.66-.6-.52-.46-1.06-.94-1.61-1.41-1.26-1.09-2-1.69-3.17-2.87Zm9.43-22.09A19.86,19.86,0,1,1,60,90.74,19.86,19.86,0,0,1,79.88,70.88Z"/></svg>',
				"description"          => esc_html__(
					"wpForo comes with built-in powerful forum SEO features. wpForo doesn't need any SEO plugin, however you should use SEO plugins for other parts of your website (posts, pages, products, etc...). wpForo disables SEO plugins on forum pages and generates all necessary SEO meta tags.",
					"wpforo"
				),
				"description_original" => "wpForo comes with built-in powerful forum SEO features. wpForo doesn't need any SEO plugin, however you should use SEO plugins for other parts of your website (posts, pages, products, etc...). wpForo disables SEO plugins on forum pages and generates all necessary SEO meta tags.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/wpforo-seo/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/seo.php' );
				},
				"options"              => [
					"seo_title"       => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Meta Titles", "wpforo" ),
						"label_original"       => "Enable eta Titles",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/wpforo-seo/#enable-meta-title-and-seo-tags",
					],
					"seo_meta"        => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Meta Tags", "wpforo" ),
						"label_original"       => "Enable Meta Tags",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/wpforo-seo/#enable-meta-title-and-seo-tags",
					],
					"seo_profile"     => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable User Profile Page indexing", "wpforo" ),
						"label_original"       => "Enable User Profile Page indexing",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/wpforo-seo/#enable-user-profile-page-indexing",
					],
					"forums_sitemap"  => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Forums Sitemap", "wpforo" ),
						"label_original"       => "Enable Forums Sitemap",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/wpforo-seo/#enable-forums-topics-and-members-sitemap",
					],
					"topics_sitemap"  => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Topics Sitemap", "wpforo" ),
						"label_original"       => "Enable Topics Sitemap",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/wpforo-seo/#enable-forums-topics-and-members-sitemap",
					],
					"members_sitemap" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Members Sitemap", "wpforo" ),
						"label_original"       => "Enable Members Sitemap",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/wpforo-seo/#enable-forums-topics-and-members-sitemap",
					],
					"dofollow"        => [
						"type"                 => 'textarea_\n',
						"placeholder"          => "example.com",
						"label"                => esc_html__( "Allowed dofollow domains", "wpforo" ),
						"label_original"       => "Allowed dofollow domains",
						"description"          => esc_html__(
							"wpForo adds nofollow to all links with external URLs. If you want to keep some domains as internal please insert domains one per line in the textarea bellow.",
							"wpforo"
						),
						"description_original" => "wpForo adds nofollow to all links with external URLs. If you want to keep some domains as internal please insert domains one per line in the textarea bellow.",
						"docurl"               => "",
					],
					"noindex"         => [
						"type"                 => 'textarea_\n',
						"placeholder"          => "https://myforum.com/community/main-forum/my-topic/",
						"label"                => esc_html__( "Noindex forum page URLs", "wpforo" ),
						"label_original"       => "Noindex forum page URLs",
						"description"          => esc_html__(
							"The noIndex code tells Google and other search engines to NOT index the page, so that it cannot be found in search results. Please insert page URLs you do not want to be indexed one per line in the textarea bellow.",
							"wpforo"
						),
						"description_original" => "The noIndex code tells Google and other search engines to NOT index the page, so that it cannot be found in search results. Please insert page URLs you do not want to be indexed one per line in the textarea bellow.",
						"docurl"               => "",
					],
				],
			],
			'antispam'      => [
				"title"                => esc_html__( "Spam Protection", "wpforo" ),
				"title_original"       => "Spam Protection",
				"icon"                 => '<svg width="55px" version="1.1" xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 111.811 122.88" enable-background="new 0 0 111.811 122.88" xml:space="preserve"><g><path fill-rule="evenodd" clip-rule="evenodd" d="M55.713,0c20.848,13.215,39.682,19.467,55.846,17.989 c2.823,57.098-18.263,90.818-55.63,104.891C19.844,109.708-1.5,77.439,0.083,17.123C19.058,18.116,37.674,14.014,55.713,0L55.713,0 z M56.163,19.543c14.217,9.011,27.061,13.274,38.083,12.268c1.925,38.936-12.454,61.93-37.935,71.526 c-0.161-0.059-0.319-0.12-0.479-0.18V19.796L56.163,19.543L56.163,19.543z M55.735,7.055 c18.454,11.697,35.126,17.232,49.434,15.923c2.498,50.541-16.166,80.39-49.241,92.846C23.986,104.165,5.091,75.603,6.493,22.211 C23.29,23.091,39.768,19.46,55.735,7.055L55.735,7.055z"/></g></svg>',
				"description"          => esc_html__(
					"Here you can find many ways to protect your forum from spammers. Most of them are already configured and enabled with default values. You can make spam protection more powerfully if you configure Google reCAPTCHA in Settings > Google reCAPTCHA section and install others spam protection plugins like Akismet.",
					"wpforo"
				),
				"description_original" => "Here you can find many ways to protect your forum from spammers. Most of them are already configured and enabled with default values. You can make spam protection more powerfully if you configure Google reCAPTCHA in Settings > Google reCAPTCHA section and install others spam protection plugins like Akismet.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/antispam.php' );
				},
				"options"              => [
					"spam_filter"                      => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable wpForo Spam Control", "wpforo" ),
						"label_original"       => "Enable wpForo Spam Control",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/#wpforo-spam-control",
					],
					"spam_user_ban"                    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Ban user when spam is suspected", "wpforo" ),
						"label_original"       => "Ban user when spam is suspected",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/#wpforo-spam-control",
					],
					"should_unapprove_after_report"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Make the reported post unapproved.", "wpforo" ),
						"label_original"       => "Make the reported post unapproved.",
						"description"          => esc_html__( "When a post is reported set it under moderation.", "wpforo" ),
						"description_original" => "When a post is reported set it under moderation.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/#wpforo-spam-control",
					],
					"spam_filter_level_topic"          => [
						"type"                 => "number",
						"min"                  => 0,
						"max"                  => 100,
						"label"                => esc_html__( "Spam Suspicion Level for Topics", "wpforo" ),
						"label_original"       => "Spam Suspicion Level for Topics",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/#wpforo-spam-control",
					],
					"spam_filter_level_post"           => [
						"type"                 => "number",
						"min"                  => 0,
						"max"                  => 100,
						"label"                => esc_html__( "Spam Suspicion Level for Posts", "wpforo" ),
						"label_original"       => "Spam Suspicion Level for Posts",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/#wpforo-spam-control",
					],
					"new_user_max_posts"               => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "User is New (under hard spam control) during first [X] posts", "wpforo" ),
						"label_original"       => "User is New (under hard spam control) during first [X] posts",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/#user-is-new-under-hard-spam-control-during-first-x-posts",
					],
					"unapprove_post_if_user_is_new"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Posts must be manually approved", "wpforo" ),
						"label_original"       => "Posts must be manually approved",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/#posts-must-be-manually-approved",
					],
					"min_number_posts_to_edit_account" => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Min number of posts to be able to edit profile information", "wpforo" ),
						"label_original"       => "Min number of posts to be able to edit profile information",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/#min-number-of-posts-to-be-able-to-edit-profile-information",
					],
					"min_number_posts_to_attach"       => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Min number of posts to be able to attach files", "wpforo" ),
						"label_original"       => "Min number of posts to be able to attach files",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/#min-number-of-posts-to-be-able-to-attach-a-file-or-post-links",
					],
					"min_number_posts_to_link"         => [
						"type"                 => "number",
						"min"                  => 0,
						"label"                => esc_html__( "Min number of posts to be able to post links", "wpforo" ),
						"label_original"       => "Min number of posts to be able to post links",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/#min-number-of-posts-to-be-able-to-attach-a-file-or-post-links",
					],
					"limited_file_ext"                 => [
						"type"                 => "textarea_|",
						"label"                => esc_html__( "Do not allow attaching files with following extensions", "wpforo" ),
						"label_original"       => "Do not allow attaching files with following extensions",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/spam-protection/#do-not-allow-attaching-files-with-following-extensions",
					],
					"spam_file_scanner"                => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable File Scanner", "wpforo" ),
						"label_original"       => "Enable File Scanner",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"exclude_file_ext"                 => [
						"type"                 => "textarea_|",
						"label"                => esc_html__( "Exclude file extensions", "wpforo" ),
						"label_original"       => "Exclude file extensions",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
				],
			],
			'recaptcha'     => [
				"title"                => esc_html__( "Google reCAPTCHA", "wpforo" ),
				"title_original"       => "Google reCAPTCHA",
				"icon"                 => '<?xml version="1.0" encoding="UTF-8"?> <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 -0.04 600 598.59" height="60px"><path d="M600 298.83c-.01-4.3-.11-8.58-.31-12.84V43.06l-67.12 67.16C477.64 42.93 394.06-.04 300.46-.04c-97.42 0-183.96 46.52-238.67 118.57l110.02 111.25a145.91 145.91 0 0 1 44.57-50.04c19.21-14.99 46.43-27.26 84.07-27.26 4.55 0 8.06.54 10.64 1.54 46.65 3.68 87.08 29.44 110.89 66.82l-77.88 77.93c98.64-.39 210.08-.62 255.89.05" fill="#2b95ce"></path><path d="M298.71-.03c-4.31.02-8.58.12-12.84.31H43.07l67.13 67.16C42.95 122.41 0 206.02 0 299.68c0 97.47 46.5 184.07 118.51 238.8L229.69 428.4a145.737 145.737 0 0 1-50-44.59c-14.99-19.22-27.25-46.45-27.25-84.12 0-4.55.53-8.07 1.53-10.65 3.69-46.67 29.43-87.12 66.79-110.94l77.88 77.92c-.39-98.7-.62-210.2.05-256.04" fill="#43A6DF"></path><path d="M.01 299.68c.02 4.3.12 8.58.31 12.84v242.93l67.12-67.16c54.94 67.29 138.51 110.26 232.12 110.26 97.41 0 183.95-46.52 238.66-118.57L428.2 368.73c-10.78 19.96-26.1 37.09-44.57 50.04-19.21 14.99-46.42 27.26-84.07 27.26-4.55 0-8.06-.54-10.64-1.54-46.64-3.68-87.08-29.44-110.88-66.82l77.88-77.93c-98.65.39-210.08.62-255.9-.05" fill="#cbcbcb"></path></svg>',
				"description"          => esc_html__(
					"reCAPTCHA protects you against spam and other types of automated abuse. It makes secure topic and post editors when Guest Posting is allowed, also it protects login and registration forms against spam attacks. wpForo comes with built-in reCAPTCHA version 2, \"I'm not a robot\" checkbox.",
					"wpforo"
				),
				"description_original" => "reCAPTCHA protects you against spam and other types of automated abuse. It makes secure topic and post editors when Guest Posting is allowed, also it protects login and registration forms against spam attacks. wpForo comes with built-in reCAPTCHA version 2, \"I'm not a robot\" checkbox.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/google-recaptcha/",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/recaptcha.php' );
				},
				"options"              => [
					"site_key_secret_key" => [
						"type"                 => "wrap",
						"label"                => esc_html__( 'reCAPTCHA API Keys', 'wpforo' ) . ' ' . __( '(version 2, "I\'m not a robot" checkbox)', 'wpforo' ),
						"label_original"       => 'reCAPTCHA API Keys (version 2, "I\'m not a robot" checkbox)',
						"description"          => __(
							                          'To start using reCAPTCHA, you need to sign up for an API key pair for your site.',
							                          'wpforo'
						                          ) . '<a href="http://www.google.com/recaptcha/admin" target="_blank">' . __( 'Register your site and get API keys here &raquo;', 'wpforo' ) . '</a>',
						"description_original" => 'To start using reCAPTCHA, you need to sign up for an API key pair for your site.',
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/google-recaptcha/#recaptcha-api-keys-version-2-%E2%80%9Ci%E2%80%99m-not-a-robot%E2%80%9D-checkbox",
						"options_keys"         => [ 'site_key', 'secret_key' ],
					],
					"site_key"            => [
						"type"                 => "text",
						"placeholder"          => "Site Key",
						"label"                => "",
						"label_original"       => "",
						"description"          => "",
						"description_original" => "",
					],
					"secret_key"          => [
						"type"                 => "text",
						"placeholder"          => "Secret Key",
						"label"                => "",
						"label_original"       => "",
						"description"          => "",
						"description_original" => "",
					],
					"theme"               => [
						"type"                 => "radio",
						"label"                => esc_html__( "reCAPTCHA Theme", "wpforo" ),
						"label_original"       => "reCAPTCHA Theme",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'light', 'label' => 'Light' ],
							[ 'value' => 'dark', 'label' => 'Dark' ],
						],
						"docurl"               => "",
					],
					"topic_editor"        => [
						"type"                 => "radio",
						"label"                => esc_html__( "Guest Topic Editor", "wpforo" ),
						"label_original"       => "Guest Topic Editor",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"post_editor"         => [
						"type"                 => "radio",
						"label"                => esc_html__( "Guest Post Editor", "wpforo" ),
						"label_original"       => "Guest Post Editor",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"wpf_login_form"      => [
						"type"                 => "radio",
						"label"                => esc_html__( "wpForo Login Form", "wpforo" ),
						"label_original"       => "wpForo Login Form",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"wpf_reg_form"        => [
						"type"                 => "radio",
						"label"                => esc_html__( "wpForo Registration Form", "wpforo" ),
						"label_original"       => "wpForo Registration Form",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"wpf_lostpass_form"   => [
						"type"                 => "radio",
						"label"                => esc_html__( "wpForo Reset Password Form", "wpforo" ),
						"label_original"       => "wpForo Reset Password Form",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"login_form"          => [
						"type"                 => "radio",
						"label"                => esc_html__( "WordPress Login Form", "wpforo" ),
						"label_original"       => "WordPress Login Form",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"reg_form"            => [
						"type"                 => "radio",
						"label"                => esc_html__( "WordPress Registration Form", "wpforo" ),
						"label_original"       => "WordPress Registration Form",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
					"lostpass_form"       => [
						"type"                 => "radio",
						"label"                => esc_html__( "WordPress Reset Password Form", "wpforo" ),
						"label_original"       => "WordPress Reset Password Form",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
				],
			],
			'akismet'       => [
				"title"                => esc_html__( "Akismet Antispam Integration", "wpforo" ),
				"title_original"       => "Akismet Antispam Integration",
				"icon"                 => '<img src="' . WPFORO_URL . '/assets/images/dashboard/akismet.png" style="height:58px;" />',
				"description"          => esc_html__(
					"Akismet checks your forum posts with authors form submissions against their global database of spam to prevent your site from publishing malicious content. Akismet is well integrated with wpForo and it makes the wpForo Spam Control more intuitive and productive.",
					"wpforo"
				),
				"description_original" => "Akismet checks your forum posts with authors form submissions against their global database of spam to prevent your site from publishing malicious content. Akismet is well integrated with wpForo and it makes the wpForo Spam Control more intuitive and productive.",
				"docurl"               => "",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/akismet.php' );
				},
				"options"              => [
					"akismet" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Akismet Integration", "wpforo" ),
						"label_original"       => "Enable Akismet Antispam Protection",
						"description"          => esc_html__( "Please make sure the Akismet plugin installed and the Akismet API keys are configured.", "wpforo" ),
						"description_original" => "",
						"docurl"               => "",
					],
				],
			],
			'rss'           => [
				"title"                => esc_html__( "Forum Feed / RSS", "wpforo" ),
				"title_original"       => "Forum Feed / RSS",
				"icon"                 => '<svg height="65px" style="enable-background:new 0 0 30.3 29.9;" version="1.1" viewBox="0 0 30.3 29.9" xml:space="preserve" xmlns="http://www.w3.org/2000/svg"><style type="text/css">.st33453{fill:none;stroke:#43a6df;stroke-width:1.25;stroke-linecap:round;stroke-linejoin:round;stroke-miterlimit:10;}.st22013{fill:none;stroke:#43a6df;stroke-width:1.1713;stroke-linecap:round;stroke-linejoin:round;stroke-miterlimit:10;}</style><g><circle class="st33453" cx="8.1" cy="20" r="2.4"/><path class="st33453" d="M5.4,13.4c0,0,9.7,0.6,9.3,9h3.4c0,0,0.5-11-12.7-12.1V13.4z"/><path class="st33453" d="M5.7,7.3c0,0,15.3,1.7,14.8,14.9h3.4c0,0,0.8-16.4-18.2-18.1V7.3z"/></g></svg>',
				"description"          => esc_html__(
					"RSS (Rich Site Summary; originally RDF Site Summary; often called Really Simple Syndication) uses a family of standard web feed formats to publish frequently updated information like new topics in forums, new posts in topics. An RSS document (called “feed”, “web feed”, or “channel”) includes topic and post  titles/links/excerpt publishing date and author’s name.",
					"wpforo"
				),
				"description_original" => "RSS (Rich Site Summary; originally RDF Site Summary; often called Really Simple Syndication) uses a family of standard web feed formats to publish frequently updated information like new topics in forums, new posts in topics. An RSS document (called “feed”, “web feed”, or “channel”) includes topic and post  titles/links/excerpt publishing date and author’s name.",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/forum-feed-rss/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/rss.php' );
				},
				"options"              => [
					"feed"         => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable RSS Feed", "wpforo" ),
						"label_original"       => "Enable RSS Feed",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/forum-feed-rss/#enable-rss-feed",
					],
					"feed_general" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable General RSS Feed", "wpforo" ),
						"label_original"       => "Enable General RSS Feed",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/forum-feed-rss/#enable-general-rss-feed",
					],
					"feed_forum"   => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Forum RSS Feed", "wpforo" ),
						"label_original"       => "Enable Forum RSS Feed",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/forum-feed-rss/#enable-forum-rss-feed",
					],
					"feed_topic"   => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Topic RSS Feed", "wpforo" ),
						"label_original"       => "Enable Topic RSS Feed",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/forum-feed-rss/#enable-topic-rss-feed",
					],
				],
			],
			'social'        => [
				"title"                => esc_html__( "Social Share", "wpforo" ),
				"title_original"       => "Share",
				"icon"                 => '<svg height="60px" data-name="Layer 1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 121.28 122.88"><path d="M50.51,46,64,37a27.93,27.93,0,1,1,6.68,10.72c-.43-.44-.85-.89-1.25-1.35L55.1,55.93a27.8,27.8,0,0,1,.76,6.5,28.37,28.37,0,0,1-.34,4.35L71.67,77.34A27.77,27.77,0,1,1,66.58,87L51.64,77.19a27.93,27.93,0,1,1-4-34.51A29.78,29.78,0,0,1,50.51,46ZM41.4,74.84a19,19,0,0,1-26.93,0L12.8,72.65,14,71.71c2.5-1.48,6.27-1.09,8.73-2.81A7.08,7.08,0,0,0,23.2,68c.23-.53.44-1.1.57-1.49a18.85,18.85,0,0,1-1.49-2.13L20.77,62a4.38,4.38,0,0,1-.86-2.2,1.69,1.69,0,0,1,.15-.79,1.42,1.42,0,0,1,.52-.6,1.55,1.55,0,0,1,.36-.19,38.65,38.65,0,0,1-.07-4.32,5.47,5.47,0,0,1,.19-1,5.78,5.78,0,0,1,2.55-3.26,8.37,8.37,0,0,1,2.14-.95c.48-.13-.41-1.67.09-1.72,2.4-.25,6.29,1.94,8,3.76a5.91,5.91,0,0,1,1.49,3.71l-.1,3.92h0a1.09,1.09,0,0,1,.8.82,3.4,3.4,0,0,1-.42,2.07h0l0,0-1.72,2.84a14.89,14.89,0,0,1-2.12,2.91l.23.33a10.22,10.22,0,0,0,1.12,1.45l0,0c2,1.41,6.81,1.75,8.67,2.78l.07,0,1.22,1a22.07,22.07,0,0,1-1.66,2.16ZM44,46.37a22.72,22.72,0,1,0,6.65,16.06A22.64,22.64,0,0,0,44,46.37Zm63.09,60.25c-5.76,6.39-21.5,6.83-27.43.52l0,0V105.8c0-3.44,5.72-3.5,8.77-5.33,2-1.21,1.68-2.43,1.67-4.44H86.85c-8.32,0-2.84.66-1.71-8.39,1.69-12.77,14.56-12.78,16.46,0C102.82,96.35,108,96,99.89,96H96.64c0,2.22-.36,3.35,1.94,4.6s8.5,1.88,8.5,5v1Zm2.33-27.73A22.72,22.72,0,1,0,116.06,95a22.64,22.64,0,0,0-6.65-16.06Zm-5.47-39.82a19,19,0,0,1-26.93,0l-1.67-2.18L76.56,36c2.5-1.48,6.27-1.09,8.73-2.81a7,7,0,0,0,.46-.88c.23-.53.43-1.1.57-1.49a18.56,18.56,0,0,1-1.5-2.13l-1.51-2.4a4.47,4.47,0,0,1-.86-2.2,1.69,1.69,0,0,1,.15-.79,1.37,1.37,0,0,1,.52-.6,1.43,1.43,0,0,1,.37-.19,36.43,36.43,0,0,1-.07-4.32,5.45,5.45,0,0,1,.18-1,5.76,5.76,0,0,1,2.56-3.26,7.9,7.9,0,0,1,2.14-1c.48-.14-.41-1.67.08-1.72,2.41-.25,6.29,1.94,8,3.76a5.82,5.82,0,0,1,1.48,3.7l-.09,3.93h0a1.1,1.1,0,0,1,.79.82,3.46,3.46,0,0,1-.41,2.07h0l0,0L96.37,28.4a14.49,14.49,0,0,1-2.12,2.91l.23.33a10.3,10.3,0,0,0,1.13,1.45s0,0,0,0c2,1.4,6.82,1.74,8.67,2.78l.08,0,1.22,1a22,22,0,0,1-1.67,2.15Zm2.59-27.2a22.72,22.72,0,1,0,6.65,16.06,22.64,22.64,0,0,0-6.65-16.06Z"/></svg>',
				"description"          => esc_html__(
					"Here you can find social network sharing options. wpForo allows sharing almost all forum content (forums, topics, posts individually). The share button style and location is very flexible. There are a lot of options to set button style, type, location, etc...",
					"wpforo"
				),
				"description_original" => "Here you can find social network sharing options. wpForo allows sharing almost all forum content (forums, topics, posts individually). The share button style and location is very flexible. There are a lot of options to set button style, type, location, etc...",
				"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/social-share/",
				"status"               => "ok",
				"base"                 => false,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/social.php' );
				},
				"options"              => [
					"sb"                 => [
						"label"                => esc_html__( "Active Share Buttons", "wpforo" ),
						"label_original"       => "Active Share Buttons",
						"description"          => esc_html__(
							"Check the checkbox share options to activate. Please note, that the Facebook share button cannot be activated without Facebook API ID. Please follow to the 'Facebook API Configuration' option instruction in 'Login & Registration' setting section and fill the API ID field in order to activate Facebook Share button.",
							"wpforo"
						),
						"description_original" => "Check the checkbox share options to activate. Please note, that the Facebook share button cannot be activated without Facebook API ID. Please follow to the 'Facebook API Configuration' option instruction in 'Login & Registration' setting section and fill the API ID field in order to activate Facebook Share button.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/social-share/#active-share-buttons",
					],
					"sb_on"              => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable General Share Buttons", "wpforo" ),
						"label_original"       => "Enable General Share Buttons",
						"description"          => esc_html__(
							"General share buttons are forum and topic sharing buttons. They are located on the top and the bottom of each page. You can manage location of these buttons using 'Share Buttons Location' options below.",
							"wpforo"
						),
						"description_original" => "General share buttons are forum and topic sharing buttons. They are located on the top and the bottom of each page. You can manage location of these buttons using 'Share Buttons Location' options below.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/social-share/#enable-general-share-buttons",
					],
					"sb_toggle_on"       => [
						"type"                 => "radio",
						"label"                => esc_html__( "Enable Post Sharing Toggle", "wpforo" ),
						"label_original"       => "Enable Post Sharing Toggle",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/social-share/#general-share-buttons-color",
					],
					"sb_style"           => [
						"type"                 => "radio",
						"label"                => esc_html__( "General Share Buttons Color", "wpforo" ),
						"label_original"       => "General Share Buttons Color",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'grey', 'label' => 'Grey' ],
							[ 'value' => 'colored', 'label' => 'Colored' ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/social-share/#general-share-buttons-color",
					],
					"sb_type"            => [
						"type"                 => "radio",
						"label"                => esc_html__( "General Share Buttons Type", "wpforo" ),
						"label_original"       => "General Share Buttons Type",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'icon', 'label' => 'Icon' ],
							[ 'value' => 'button', 'label' => 'Button' ],
							[ 'value' => 'button_count', 'label' => 'Counter' ],
						],
						"docurl"               => "",
					],
					"sb_toggle"          => [
						"label"                => esc_html__( "Post Sharing Toggle", "wpforo" ),
						"label_original"       => "Post Sharing Toggle",
						"description"          => esc_html__(
							"Post sharing toggle allows you to share posts individually. You can see post sharing toggles on the left, right side or in top bar of each post. The toggle blue color is the current primary color (#12) of your forum style. For example, if you use the red forum style, the color of all share toggles will be red. This doesn't affect share button colors. They are always gray with original colors on mouse hover.",
							"wpforo"
						),
						"description_original" => "Post sharing toggle allows you to share posts individually. You can see post sharing toggles on the left, right side or in top bar of each post. The toggle blue color is the current primary color (#12) of your forum style. For example, if you use the red forum style, the color of all share toggles will be red. This doesn't affect share button colors. They are always gray with original colors on mouse hover.",
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/social-share/#post-sharing-toggle",
					],
					"sb_toggle_type"     => [
						"type"                 => "radio",
						"label"                => esc_html__( "Post Sharing Toggle View", "wpforo" ),
						"label_original"       => "Post Sharing Toggle View",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'collapsed', 'label' => __( 'Collapsed', 'wpforo' ) ],
							[ 'value' => 'expanded', 'label' => __( 'Expanded', 'wpforo' ) ],
						],
						"docurl"               => "https://wpforo.com/docs/wpforo-v2/settings/social-share/#post-sharing-toggle",
					],
					"sb_icon"            => [
						"type"                 => "radio",
						"label"                => esc_html__( "Post Sharing Toggle Icon", "wpforo" ),
						"label_original"       => "Post Sharing Toggle Icon",
						"description"          => esc_html__( "", "wpforo" ),
						"description_original" => "",
						"variants"             => [
							[ 'value' => 'mixed', 'label' => 'Mixed' ],
							[ 'value' => 'figure', 'label' => '<i class="fab fa-facebook-f" style="font-size: 13px;"></i> Figure' ],
							[ 'value' => 'square', 'label' => '<i class="fab fa-facebook-square" style="font-size: 14px;"></i> Square' ],
						],
						"docurl"               => "",
					],
					"sb_location"        => [
						"label"                => esc_html__( "General Share Button Locations", "wpforo" ),
						"label_original"       => "General Share Button Locations",
						"description"          => esc_html__( "The general share buttons can be displayed on both (top and bottom) locations.", "wpforo" ),
						"description_original" => "The general share buttons can be displayed on both (top and bottom) locations.",
						"docurl"               => "",
					],
					"sb_location_toggle" => [
						"label"                => esc_html__( "Post Sharing Toggle Locations", "wpforo" ),
						"label_original"       => "Post Sharing Toggle Locations",
						"description"          => esc_html__( "The post sharing toggle can be displayed either on the left side or on the top of each post.", "wpforo" ),
						"description_original" => "The post sharing toggle can be displayed either on the left side or on the top of each post.",
						"docurl"               => "",
					],
				],
			],
			'buddypress'    => [
				"title"                => esc_html__( "BuddyPress Integration", "wpforo" ),
				"title_original"       => "BuddyPress Integration",
				"icon"                 => '<svg height="60px" xmlns="http://www.w3.org/2000/svg" aria-hidden="true" role="img" preserveAspectRatio="xMidYMid meet" viewBox="0 0 20 20"><path d="M10 0c5.52 0 10 4.48 10 10s-4.48 10-10 10S0 15.52 0 10S4.48 0 10 0zm0 .5C4.75.5.5 4.75.5 10s4.25 9.5 9.5 9.5s9.5-4.25 9.5-9.5S15.25.5 10 .5zm0 1c4.7 0 8.5 3.8 8.5 8.5s-3.8 8.5-8.5 8.5s-8.5-3.8-8.5-8.5S5.3 1.5 10 1.5zm1.8 1.71c-.57 0-1.1.17-1.55.45a3.55 3.55 0 0 1 2.73 3.45c0 .69-.21 1.33-.55 1.87a2.917 2.917 0 0 0 2.29-2.85c0-1.61-1.31-2.92-2.92-2.92zm-2.38 1a2.926 2.926 0 1 0 .011 5.851A2.926 2.926 0 0 0 9.42 4.21zm4.25 5.01l-.51.59c2.34.69 2.45 3.61 2.45 3.61h1.28c0-4.71-3.22-4.2-3.22-4.2zm-2.1.8l-2.12 2.09l-2.12-2.09C3.12 10.24 3.89 15 3.89 15h11.08c.47-4.98-3.4-4.98-3.4-4.98z" fill="currentColor"/></svg>',
				"description"          => esc_html__(
					"wpForo is well integrated with BuddyPress and with all plugins and themes based on BuddyPress. The integration includes changing forum profile page URL to BuddyPress profile page URL, Activity Integration and Forum Tabs on BuddyPress profile page.",
					"wpforo"
				),
				"description_original" => "wpForo is well integrated with BuddyPress and with all plugins and themes based on BuddyPress. The integration includes changing forum profile page URL to BuddyPress profile page URL, Activity Integration and Forum Tabs on BuddyPress profile page.",
				"docurl"               => "",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/buddypress.php' );
				},
				"options"              => [
					"activity"     => [
						"type"                 => "radio",
						"label"                => esc_html__( "BuddyPress Activity Integration", "wpforo" ),
						"label_original"       => "BuddyPress Activity Integration",
						"description"          => esc_html__( "Posts members activity (new topic, new reply, post like) to BuddyPress Profile Activity page.", "wpforo" ),
						"description_original" => "Posts members activity (new topic, new reply, post like) to BuddyPress Profile Activity page.",
						"docurl"               => "",
					],
					"notification" => [
						"type"                 => "radio",
						"label"                => esc_html__( "BuddyPress Notification Integration", "wpforo" ),
						"label_original"       => "BuddyPress Notification Integration",
						"description"          => esc_html__( "Creates notification on new forum reply in BuddyPress Profile Notification page.", "wpforo" ),
						"description_original" => "Creates notification on new forum reply in BuddyPress Profile Notification page.",
						"docurl"               => "",
					],
					"forum_tab"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "BuddyPress Profile Forum Tab Integration", "wpforo" ),
						"label_original"       => "BuddyPress Profile Forum Tab Integration",
						"description"          => esc_html__(
							"Adds 'Forums' tab with 'Created Topics', 'Posted Replies', 'Liked Posts' and 'Subscriptions' sub-tabs to BuddyPress Profile page.",
							"wpforo"
						),
						"description_original" => "Adds 'Forums' tab with 'Created Topics', 'Posted Replies', 'Liked Posts' and 'Subscriptions' sub-tabs to BuddyPress Profile page.",
						"docurl"               => "",
					],
				],
			],
			'um'            => [
				"title"                => esc_html__( "Ultimate Members Integration", "wpforo" ),
				"title_original"       => "Ultimate Members Integration",
				"icon"                 => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" height="60px" viewBox="0, 0, 400,400"><g id="svgg"><path id="path0" d="M114.054 7.863 C 113.534 7.976,112.544 8.412,111.854 8.832 C 111.164 9.253,110.187 9.598,109.683 9.599 C 109.179 9.599,108.380 9.960,107.908 10.400 C 107.436 10.840,106.737 11.200,106.355 11.200 C 105.974 11.200,105.127 11.650,104.473 12.200 C 103.819 12.750,103.122 13.200,102.923 13.200 C 102.724 13.200,102.329 13.842,102.045 14.626 C 101.761 15.410,101.275 16.150,100.965 16.269 C 100.654 16.388,100.400 16.701,100.400 16.965 C 100.400 17.229,99.950 18.055,99.400 18.800 C 98.815 19.593,98.400 20.697,98.400 21.459 C 98.400 22.873,97.673 24.800,97.140 24.800 C 96.958 24.800,96.762 48.935,96.704 78.432 C 96.606 128.939,96.559 132.094,95.901 132.575 C 95.517 132.856,95.091 133.646,94.954 134.331 C 94.810 135.051,94.387 135.677,93.952 135.815 C 93.539 135.946,93.200 136.333,93.200 136.674 C 93.200 137.369,90.864 139.584,90.120 139.594 C 89.856 139.597,89.464 139.915,89.248 140.300 C 88.886 140.947,87.539 141.009,71.410 141.116 C 56.527 141.215,53.929 141.318,53.738 141.818 C 53.614 142.141,52.812 142.517,51.956 142.653 C 50.807 142.837,50.400 143.093,50.400 143.633 C 50.400 144.035,49.950 144.814,49.400 145.364 C 48.850 145.914,48.400 146.763,48.400 147.250 C 48.400 147.738,48.085 148.485,47.700 148.911 C 46.586 150.142,46.619 249.006,47.735 251.333 C 48.266 252.442,48.525 254.159,48.671 257.533 C 48.794 260.387,49.104 262.715,49.467 263.526 C 49.794 264.255,50.244 266.671,50.467 268.895 C 50.690 271.118,51.125 273.325,51.435 273.798 C 51.750 274.278,52.114 276.277,52.259 278.329 C 52.438 280.849,52.676 282.000,53.017 282.000 C 53.631 282.000,54.000 282.904,54.000 284.410 C 54.000 285.065,54.315 285.778,54.736 286.077 C 55.197 286.405,55.571 287.345,55.738 288.600 C 55.886 289.709,56.292 290.813,56.649 291.078 C 57.004 291.342,57.385 292.370,57.497 293.367 C 57.645 294.674,57.910 295.245,58.451 295.416 C 58.947 295.574,59.200 296.030,59.200 296.769 C 59.200 297.403,59.631 298.348,60.200 298.962 C 60.771 299.578,61.200 300.521,61.200 301.159 C 61.200 301.773,61.542 302.586,61.960 302.964 C 62.379 303.343,62.839 304.284,62.984 305.055 C 63.129 305.827,63.596 307.030,64.022 307.729 C 64.448 308.428,64.798 309.315,64.799 309.700 C 64.799 310.085,65.160 310.760,65.600 311.200 C 66.040 311.640,66.400 312.354,66.400 312.786 C 66.400 313.218,66.760 313.765,67.200 314.000 C 67.640 314.235,68.000 314.683,68.000 314.993 C 68.000 315.304,68.450 316.044,69.000 316.638 C 69.550 317.231,70.000 318.042,70.000 318.440 C 70.000 318.838,70.405 319.608,70.900 320.151 C 72.553 321.964,73.600 323.520,73.600 324.163 C 73.600 324.513,73.868 324.800,74.195 324.800 C 74.522 324.800,75.034 325.385,75.333 326.100 C 75.632 326.815,76.174 327.647,76.538 327.949 C 76.902 328.251,77.200 328.905,77.200 329.402 C 77.200 329.960,77.497 330.384,77.974 330.509 C 78.478 330.641,78.839 331.202,79.011 332.119 C 79.182 333.027,79.546 333.597,80.038 333.726 C 80.457 333.835,80.800 334.200,80.800 334.536 C 80.800 334.871,81.160 335.260,81.600 335.400 C 82.040 335.540,82.400 335.916,82.400 336.237 C 82.400 336.557,82.850 336.991,83.400 337.200 C 83.950 337.409,84.400 337.855,84.400 338.190 C 84.400 338.526,84.674 338.800,85.010 338.800 C 85.345 338.800,85.791 339.250,86.000 339.800 C 86.209 340.350,86.569 340.800,86.800 340.800 C 87.031 340.800,87.391 341.250,87.600 341.800 C 87.809 342.350,88.255 342.800,88.590 342.800 C 88.926 342.800,89.200 343.064,89.200 343.386 C 89.200 343.708,89.740 344.587,90.400 345.339 C 91.060 346.091,91.600 346.907,91.600 347.153 C 91.600 347.399,91.844 347.600,92.142 347.600 C 92.439 347.600,93.154 348.036,93.729 348.569 C 94.304 349.102,95.140 349.654,95.587 349.796 C 96.034 349.938,96.400 350.316,96.400 350.637 C 96.400 350.957,96.850 351.391,97.400 351.600 C 97.950 351.809,98.400 352.255,98.400 352.590 C 98.400 352.926,98.589 353.200,98.820 353.200 C 99.396 353.200,102.000 355.772,102.000 356.340 C 102.000 356.593,102.360 356.800,102.800 356.800 C 103.318 356.800,103.600 357.067,103.600 357.557 C 103.600 358.030,103.974 358.407,104.600 358.564 C 105.246 358.726,105.600 359.096,105.600 359.608 C 105.600 360.212,105.870 360.400,106.742 360.400 C 107.767 360.400,108.332 360.752,110.900 362.990 C 111.285 363.325,111.754 363.600,111.942 363.600 C 112.129 363.600,112.769 364.050,113.362 364.600 C 113.956 365.150,114.719 365.600,115.058 365.600 C 115.397 365.600,115.770 365.962,115.885 366.404 C 116.001 366.847,116.511 367.313,117.018 367.440 C 117.526 367.568,118.252 368.016,118.633 368.436 C 119.013 368.856,119.568 369.200,119.866 369.200 C 120.164 369.200,120.716 369.560,121.093 370.000 C 121.470 370.440,121.982 370.800,122.230 370.800 C 122.479 370.800,123.169 371.250,123.762 371.800 C 124.356 372.350,125.190 372.800,125.617 372.800 C 126.044 372.800,126.618 373.160,126.893 373.600 C 127.186 374.069,127.826 374.400,128.440 374.400 C 129.017 374.400,130.151 374.850,130.962 375.400 C 131.772 375.950,132.760 376.400,133.156 376.400 C 133.552 376.400,134.182 376.738,134.556 377.152 C 134.931 377.565,135.904 378.010,136.720 378.141 C 137.536 378.271,138.516 378.743,138.898 379.189 C 139.309 379.669,140.082 380.000,140.793 380.000 C 141.560 380.000,142.174 380.289,142.493 380.800 C 142.790 381.276,143.426 381.600,144.064 381.600 C 145.519 381.600,146.400 381.980,146.400 382.608 C 146.400 382.910,147.115 383.250,148.100 383.416 C 149.035 383.573,150.229 384.039,150.754 384.451 C 151.281 384.865,152.355 385.202,153.154 385.205 C 155.192 385.213,156.400 385.601,156.400 386.247 C 156.400 386.654,157.011 386.800,158.715 386.800 C 160.262 386.800,161.557 387.071,162.615 387.616 C 163.762 388.207,165.306 388.496,168.200 388.663 C 170.975 388.822,172.588 389.114,173.468 389.614 C 174.462 390.180,175.998 390.386,180.567 390.571 C 184.751 390.739,186.878 391.002,188.099 391.503 C 190.621 392.537,211.921 392.558,213.059 391.528 C 213.648 390.995,214.999 390.799,219.581 390.584 C 223.609 390.395,225.633 390.134,226.260 389.724 C 226.754 389.400,228.652 388.983,230.479 388.796 C 232.306 388.609,234.610 388.154,235.600 387.784 C 236.590 387.413,238.982 387.015,240.916 386.899 C 243.800 386.725,244.504 386.553,244.830 385.944 C 245.114 385.413,245.677 385.200,246.798 385.200 C 247.807 385.200,248.625 384.934,249.084 384.456 C 249.478 384.046,250.598 383.589,251.574 383.439 C 252.651 383.274,253.564 382.859,253.897 382.383 C 254.285 381.829,254.910 381.600,256.035 381.600 C 257.074 381.600,257.704 381.394,257.854 381.004 C 257.979 380.677,258.793 380.217,259.662 379.983 C 260.530 379.749,261.339 379.302,261.459 378.990 C 261.579 378.677,262.335 378.299,263.139 378.148 C 263.942 377.998,265.050 377.560,265.600 377.175 C 266.150 376.790,267.275 376.360,268.100 376.219 C 268.925 376.079,269.600 375.821,269.600 375.646 C 269.600 375.238,271.431 374.400,272.323 374.400 C 272.699 374.400,273.232 374.040,273.507 373.600 C 273.782 373.160,274.356 372.800,274.783 372.800 C 275.210 372.800,276.044 372.350,276.638 371.800 C 277.231 371.250,277.871 370.800,278.058 370.800 C 278.246 370.800,278.760 370.440,279.200 370.000 C 279.640 369.560,280.467 369.200,281.037 369.200 C 281.738 369.200,282.143 368.941,282.284 368.400 C 282.399 367.960,282.744 367.600,283.049 367.600 C 283.355 367.600,283.997 367.150,284.475 366.600 C 284.953 366.050,285.582 365.600,285.872 365.600 C 286.162 365.600,286.400 365.459,286.400 365.286 C 286.400 365.114,286.985 364.633,287.700 364.218 C 288.415 363.802,289.140 363.231,289.311 362.947 C 289.482 362.664,290.241 362.316,290.997 362.174 C 291.847 362.015,292.465 361.626,292.613 361.158 C 292.746 360.741,293.195 360.400,293.613 360.400 C 294.030 360.400,294.565 360.040,294.800 359.600 C 295.035 359.160,295.497 358.800,295.827 358.800 C 296.156 358.800,296.542 358.485,296.685 358.100 C 296.978 357.310,301.671 352.680,303.200 351.673 C 303.750 351.310,304.435 350.696,304.723 350.307 C 305.010 349.918,305.583 349.600,305.996 349.600 C 306.408 349.600,306.860 349.240,307.000 348.800 C 307.140 348.360,307.516 348.000,307.837 348.000 C 308.157 348.000,308.591 347.550,308.800 347.000 C 309.009 346.450,309.455 346.000,309.790 346.000 C 310.126 346.000,310.405 345.685,310.411 345.300 C 310.425 344.388,316.420 337.600,317.211 337.600 C 317.425 337.600,317.600 337.334,317.600 337.009 C 317.600 336.684,318.050 336.124,318.600 335.763 C 319.150 335.403,319.600 334.842,319.600 334.517 C 319.600 334.191,319.960 333.831,320.400 333.716 C 320.840 333.601,321.200 333.239,321.200 332.911 C 321.200 332.584,321.650 331.781,322.200 331.127 C 322.750 330.473,323.200 329.689,323.200 329.383 C 323.200 329.078,323.560 328.635,324.000 328.400 C 324.440 328.165,324.800 327.742,324.800 327.461 C 324.800 327.181,325.138 326.588,325.551 326.145 C 325.964 325.701,326.426 324.947,326.578 324.469 C 326.730 323.991,327.202 323.413,327.627 323.186 C 328.052 322.958,328.400 322.428,328.400 322.009 C 328.400 321.589,328.734 321.010,329.143 320.723 C 329.551 320.435,330.019 319.777,330.182 319.261 C 330.345 318.744,330.821 317.979,331.239 317.561 C 331.658 317.142,332.000 316.542,332.000 316.227 C 332.000 315.912,332.360 315.540,332.800 315.400 C 333.240 315.260,333.600 314.798,333.600 314.373 C 333.600 313.948,333.957 313.243,334.394 312.806 C 334.831 312.369,335.302 311.300,335.442 310.430 C 335.581 309.561,336.003 308.518,336.380 308.113 C 336.758 307.708,337.181 306.672,337.321 305.811 C 337.460 304.950,337.836 304.100,338.156 303.921 C 338.476 303.742,338.853 303.071,338.993 302.431 C 339.134 301.790,339.598 300.863,340.025 300.372 C 340.451 299.880,340.800 299.055,340.800 298.539 C 340.800 298.022,341.160 297.240,341.600 296.800 C 342.040 296.360,342.400 295.577,342.400 295.060 C 342.400 293.661,343.177 291.695,343.828 291.445 C 344.172 291.313,344.400 290.671,344.400 289.835 C 344.400 288.942,344.663 288.259,345.136 287.923 C 345.622 287.578,345.957 286.671,346.123 285.254 C 346.264 284.042,346.640 282.910,346.986 282.654 C 347.405 282.344,347.676 281.187,347.842 279.000 C 347.976 277.240,348.407 275.080,348.801 274.200 C 349.238 273.221,349.600 271.161,349.733 268.892 C 349.865 266.617,350.205 264.685,350.612 263.892 C 351.109 262.923,351.357 260.822,351.601 255.486 C 351.849 250.075,352.079 248.138,352.563 247.400 C 353.662 245.723,353.522 148.793,352.420 148.105 C 351.991 147.837,351.524 147.000,351.382 146.246 C 351.218 145.369,350.849 144.802,350.362 144.674 C 349.919 144.558,349.600 144.125,349.600 143.637 C 349.600 143.068,349.345 142.800,348.805 142.800 C 348.367 142.800,347.641 142.485,347.192 142.100 C 346.449 141.464,344.792 141.382,328.955 141.200 C 312.963 141.016,311.493 140.943,311.023 140.300 C 310.742 139.915,310.313 139.600,310.070 139.600 C 309.449 139.600,306.800 137.214,306.800 136.655 C 306.800 136.400,306.440 135.968,306.000 135.693 C 305.560 135.418,305.200 134.953,305.200 134.659 C 305.200 134.365,304.844 133.802,304.410 133.409 C 303.631 132.704,303.618 131.893,303.510 78.389 C 303.408 27.902,303.352 24.031,302.719 23.331 C 302.345 22.918,301.918 21.774,301.772 20.790 C 301.625 19.805,301.240 18.788,300.916 18.530 C 300.316 18.050,299.229 16.274,298.327 14.300 C 298.051 13.695,297.604 13.200,297.335 13.200 C 297.066 13.200,296.602 12.853,296.305 12.428 C 296.007 12.004,295.265 11.531,294.654 11.378 C 294.043 11.225,293.333 10.762,293.076 10.350 C 292.818 9.937,292.213 9.600,291.730 9.600 C 290.589 9.600,288.693 8.873,288.338 8.300 C 287.947 7.668,280.868 7.645,280.478 8.275 C 280.081 8.916,278.200 9.600,276.836 9.600 C 276.224 9.600,275.398 9.960,275.000 10.400 C 274.602 10.840,274.035 11.200,273.741 11.200 C 272.812 11.200,269.421 14.027,268.899 15.238 C 268.623 15.877,268.247 16.400,268.062 16.400 C 267.878 16.400,267.511 17.120,267.247 18.000 C 266.984 18.880,266.531 19.600,266.241 19.600 C 265.889 19.600,265.628 20.429,265.456 22.100 C 265.281 23.791,264.971 24.767,264.499 25.117 C 263.840 25.605,263.797 28.697,263.754 78.764 L 263.707 131.893 262.854 132.746 C 262.384 133.216,262.000 133.958,262.000 134.397 C 262.000 134.835,261.640 135.418,261.200 135.693 C 260.760 135.968,260.400 136.422,260.400 136.702 C 260.400 137.253,258.028 139.569,257.444 139.588 C 257.248 139.595,256.858 139.915,256.576 140.300 C 255.819 141.336,144.939 141.393,143.742 140.358 C 143.334 140.005,142.415 139.466,141.700 139.161 C 140.985 138.855,140.400 138.459,140.400 138.281 C 140.400 138.102,139.861 137.381,139.202 136.678 C 138.543 135.975,138.003 135.097,138.002 134.727 C 138.001 134.357,137.642 133.940,137.204 133.801 C 136.416 133.551,136.407 133.055,136.304 78.816 C 136.209 28.921,136.142 24.018,135.549 23.342 C 135.191 22.934,134.775 21.748,134.625 20.707 C 134.476 19.665,134.113 18.675,133.820 18.507 C 133.526 18.338,132.992 17.570,132.632 16.800 C 131.457 14.284,130.856 13.438,130.022 13.130 C 129.570 12.963,129.200 12.550,129.200 12.213 C 129.200 11.876,128.866 11.600,128.458 11.600 C 128.051 11.600,127.231 11.150,126.638 10.600 C 125.938 9.952,125.093 9.600,124.233 9.600 C 123.497 9.600,122.438 9.231,121.851 8.769 C 120.774 7.921,116.229 7.393,114.054 7.863 M106.445 218.950 C 107.020 219.250,108.860 219.624,110.533 219.781 C 112.483 219.964,113.904 220.322,114.488 220.778 C 114.990 221.170,115.822 221.622,116.338 221.784 C 120.874 223.206,128.800 232.410,128.800 236.255 C 128.800 236.850,129.115 237.684,129.500 238.109 C 130.618 239.342,130.773 249.697,129.696 251.159 C 129.275 251.732,128.810 252.959,128.663 253.886 C 128.411 255.474,127.231 257.725,125.825 259.300 C 125.481 259.685,125.200 260.171,125.200 260.379 C 125.200 261.318,120.754 265.600,119.779 265.600 C 119.461 265.600,119.200 265.761,119.200 265.958 C 119.200 266.606,115.301 268.834,113.735 269.081 C 112.891 269.214,111.573 269.655,110.806 270.062 C 108.768 271.141,100.563 271.130,98.226 270.045 C 97.332 269.630,95.732 269.084,94.672 268.832 C 93.611 268.580,92.012 267.885,91.118 267.287 C 90.224 266.689,89.086 265.926,88.590 265.592 C 87.256 264.693,83.627 260.767,82.959 259.500 C 82.640 258.895,82.227 258.400,82.040 258.400 C 81.853 258.400,81.590 257.850,81.455 257.177 C 81.321 256.504,80.945 255.688,80.620 255.363 C 79.740 254.483,79.205 252.128,79.202 249.124 C 79.201 247.620,79.008 246.329,78.762 246.177 C 78.504 246.017,78.416 245.312,78.548 244.453 C 78.671 243.654,78.963 241.380,79.196 239.400 C 79.481 236.972,79.876 235.447,80.410 234.716 C 80.844 234.120,81.200 233.416,81.200 233.151 C 81.200 230.539,90.179 221.600,92.804 221.600 C 93.097 221.600,93.658 221.310,94.049 220.955 C 94.462 220.582,96.074 220.142,97.881 219.909 C 99.596 219.688,101.360 219.362,101.800 219.183 C 104.070 218.262,105.034 218.214,106.445 218.950 M301.541 222.942 C 302.938 223.548,304.288 224.223,304.541 224.441 C 304.793 224.659,305.900 225.507,307.000 226.324 C 310.014 228.564,313.600 232.560,313.600 233.680 C 313.600 233.970,313.937 234.418,314.350 234.676 C 314.762 234.933,315.226 235.651,315.381 236.272 C 315.536 236.892,315.904 237.670,316.198 238.000 C 318.217 240.259,318.054 253.655,315.967 257.047 C 315.545 257.733,315.200 258.587,315.200 258.944 C 315.200 259.301,314.840 259.818,314.400 260.093 C 313.960 260.368,313.600 260.898,313.600 261.272 C 313.600 262.119,307.105 268.652,305.892 269.025 C 305.401 269.176,304.678 269.637,304.284 270.050 C 303.891 270.462,303.060 270.800,302.438 270.800 C 301.817 270.800,300.793 271.205,300.164 271.700 C 298.435 273.059,284.880 273.089,283.000 271.737 C 282.340 271.263,281.125 270.760,280.300 270.619 C 279.475 270.479,278.800 270.194,278.800 269.987 C 278.800 269.780,278.215 269.359,277.500 269.052 C 275.459 268.175,269.600 262.082,269.600 260.837 C 269.600 260.597,269.240 260.040,268.800 259.600 C 268.360 259.160,268.000 258.464,268.000 258.054 C 268.000 257.644,267.661 256.877,267.247 256.351 C 265.391 253.991,265.385 240.188,267.240 238.174 C 267.658 237.720,268.000 236.955,268.000 236.474 C 268.000 235.993,268.360 235.240,268.800 234.800 C 269.240 234.360,269.600 233.794,269.600 233.542 C 269.600 233.290,269.971 232.705,270.424 232.242 C 270.877 231.779,271.872 230.612,272.636 229.648 C 274.992 226.674,280.770 222.892,284.298 222.015 C 287.777 221.149,298.774 221.741,301.541 222.942 M125.400 283.200 C 126.280 283.625,127.223 283.979,127.497 283.986 C 127.770 283.994,128.218 284.360,128.493 284.800 C 128.768 285.240,129.155 285.600,129.354 285.600 C 129.553 285.600,130.095 285.971,130.558 286.424 C 132.842 288.659,136.416 291.859,138.680 293.695 C 140.066 294.820,141.200 295.888,141.200 296.070 C 141.200 296.251,141.460 296.400,141.777 296.400 C 142.095 296.400,142.591 296.715,142.880 297.100 C 143.169 297.485,144.485 298.337,145.803 298.993 C 147.121 299.649,148.560 300.481,149.000 300.841 C 149.822 301.515,151.207 302.299,153.517 303.398 C 154.242 303.743,154.929 304.272,155.045 304.573 C 155.161 304.875,155.882 305.239,156.649 305.383 C 157.415 305.527,158.142 305.904,158.264 306.222 C 158.386 306.540,158.838 306.800,159.269 306.800 C 160.246 306.800,161.697 307.542,162.120 308.258 C 162.296 308.556,163.061 308.800,163.820 308.800 C 164.634 308.800,165.506 309.106,165.947 309.547 C 166.357 309.957,167.557 310.423,168.613 310.581 C 169.669 310.739,170.908 311.110,171.367 311.406 C 171.825 311.701,173.370 312.144,174.800 312.391 C 176.230 312.638,178.436 313.191,179.702 313.620 C 180.967 314.049,182.861 314.400,183.910 314.400 C 184.958 314.400,187.115 314.760,188.703 315.200 C 191.977 316.107,212.808 316.374,216.400 315.555 C 217.390 315.329,220.075 314.890,222.367 314.579 C 224.659 314.269,227.269 313.747,228.167 313.420 C 229.065 313.093,230.866 312.640,232.169 312.414 C 233.472 312.187,234.937 311.740,235.425 311.420 C 235.913 311.101,237.277 310.716,238.456 310.565 C 239.734 310.402,240.889 309.990,241.316 309.546 C 241.741 309.102,242.587 308.800,243.402 308.800 C 244.343 308.800,244.900 308.560,245.183 308.033 C 245.409 307.610,246.180 307.140,246.897 306.988 C 247.614 306.836,248.470 306.479,248.800 306.195 C 249.130 305.912,250.030 305.392,250.800 305.041 C 251.570 304.689,252.740 304.042,253.400 303.603 C 254.060 303.164,255.323 302.468,256.206 302.057 C 257.089 301.646,258.056 301.015,258.355 300.655 C 258.653 300.295,259.148 300.000,259.453 300.000 C 259.758 300.000,260.316 299.640,260.693 299.200 C 261.070 298.760,261.582 298.400,261.830 298.400 C 262.079 298.400,262.769 297.950,263.362 297.400 C 263.956 296.850,264.716 296.400,265.052 296.400 C 265.388 296.400,266.434 295.635,267.376 294.700 C 268.318 293.765,270.124 292.100,271.388 291.000 C 272.653 289.900,274.028 288.685,274.444 288.300 C 274.860 287.915,275.373 287.600,275.584 287.600 C 275.796 287.600,276.341 287.201,276.796 286.712 C 278.658 284.714,286.696 285.890,286.785 288.174 C 286.793 288.380,287.180 288.953,287.645 289.448 C 288.974 290.863,288.836 295.153,287.405 296.907 C 284.768 300.139,284.177 300.783,283.842 300.791 C 283.488 300.800,282.992 301.219,280.540 303.584 C 279.846 304.253,279.190 304.800,279.082 304.800 C 278.973 304.800,278.295 305.350,277.575 306.023 C 275.760 307.719,272.252 310.000,271.459 310.000 C 271.096 310.000,270.800 310.276,270.800 310.613 C 270.800 310.949,270.485 311.341,270.100 311.483 C 269.715 311.624,269.004 312.159,268.519 312.670 C 268.035 313.182,267.310 313.600,266.908 313.600 C 266.506 313.600,265.870 313.960,265.493 314.400 C 265.116 314.840,264.558 315.200,264.253 315.200 C 263.948 315.200,263.451 315.498,263.149 315.862 C 262.847 316.226,262.060 316.776,261.400 317.084 C 260.740 317.392,258.868 318.354,257.240 319.222 C 255.612 320.090,253.971 320.800,253.594 320.800 C 253.216 320.800,252.479 321.132,251.954 321.539 C 251.429 321.945,250.306 322.387,249.457 322.521 C 248.609 322.654,247.708 323.046,247.456 323.391 C 247.204 323.736,246.278 324.129,245.399 324.263 C 244.519 324.398,243.499 324.769,243.131 325.088 C 242.763 325.407,241.683 325.769,240.731 325.894 C 239.779 326.018,238.571 326.453,238.046 326.860 C 237.497 327.286,236.441 327.600,235.558 327.600 C 234.715 327.600,233.326 327.964,232.472 328.410 C 231.600 328.864,229.707 329.327,228.159 329.464 C 226.642 329.599,225.150 329.907,224.845 330.149 C 223.578 331.153,203.084 332.637,198.419 332.063 C 196.538 331.831,192.480 331.439,189.400 331.191 C 186.320 330.944,183.170 330.496,182.400 330.195 C 181.630 329.895,179.920 329.535,178.600 329.396 C 177.280 329.257,175.519 328.796,174.688 328.372 C 173.856 327.947,172.416 327.599,171.488 327.597 C 170.477 327.596,169.319 327.276,168.600 326.800 C 167.940 326.363,166.929 326.004,166.354 326.003 C 165.779 326.001,164.879 325.663,164.354 325.251 C 163.829 324.839,162.622 324.371,161.672 324.211 C 160.722 324.051,159.849 323.673,159.734 323.371 C 159.618 323.069,158.865 322.713,158.061 322.581 C 156.125 322.261,154.400 321.588,154.400 321.151 C 154.400 320.958,154.112 320.800,153.759 320.800 C 153.406 320.800,152.281 320.366,151.259 319.836 C 150.237 319.305,148.950 318.666,148.400 318.415 C 146.216 317.417,144.252 316.309,143.711 315.769 C 143.399 315.456,142.892 315.200,142.586 315.200 C 142.279 315.200,141.835 314.840,141.600 314.400 C 141.365 313.960,140.837 313.600,140.428 313.600 C 140.019 313.600,139.260 313.153,138.742 312.607 C 138.224 312.061,137.575 311.611,137.300 311.607 C 137.025 311.603,136.458 311.285,136.040 310.900 C 135.623 310.515,134.619 309.750,133.810 309.200 C 133.001 308.650,131.876 307.795,131.311 307.300 C 130.746 306.805,130.138 306.400,129.960 306.400 C 129.782 306.400,129.185 305.995,128.633 305.500 C 128.080 305.005,127.077 304.150,126.403 303.600 C 124.641 302.162,119.215 296.829,117.952 295.293 C 115.526 292.345,116.586 284.000,119.385 284.000 C 119.547 284.000,120.246 283.746,120.940 283.436 C 122.243 282.852,123.253 282.476,123.600 282.445 C 123.710 282.435,124.520 282.775,125.400 283.200 " stroke="none" fill="#43A6DF" fill-rule="evenodd"></path><path id="path1" d="M0.000 200.000 L 0.000 400.000 200.000 400.000 L 400.000 400.000 400.000 200.000 L 400.000 0.000 200.000 0.000 L 0.000 0.000 0.000 200.000 M124.915 7.704 C 129.168 9.661,132.784 13.081,134.954 17.196 C 138.058 23.082,138.108 23.782,137.829 57.200 C 137.402 108.391,137.452 129.673,138.002 131.000 C 139.110 133.674,139.405 134.229,140.064 134.894 C 140.443 135.275,140.899 135.815,141.077 136.094 C 143.359 139.668,142.104 139.587,199.630 139.869 L 250.660 140.120 253.595 139.229 C 257.669 137.992,260.204 135.439,261.500 131.269 L 262.400 128.374 262.445 93.687 C 262.517 38.126,262.826 23.112,263.950 20.400 C 267.676 11.412,274.485 6.688,283.800 6.630 C 290.267 6.590,293.059 7.550,297.153 11.224 C 297.536 11.568,298.287 12.156,298.823 12.531 C 300.694 13.842,303.955 20.135,304.549 23.584 C 304.867 25.431,305.003 41.371,305.009 77.800 C 305.018 134.816,304.883 131.769,307.557 135.146 C 310.973 139.459,311.853 139.638,331.208 139.947 L 347.016 140.200 349.245 141.679 C 352.126 143.592,353.511 145.449,354.508 148.740 C 356.035 153.777,355.039 248.948,353.348 259.600 C 351.393 271.921,351.218 272.766,349.348 278.925 C 348.827 280.643,348.400 282.429,348.400 282.893 C 348.400 283.681,347.085 287.368,345.772 290.260 C 345.457 290.954,345.200 291.721,345.200 291.965 C 345.200 292.209,344.864 293.172,344.454 294.104 C 344.044 295.037,343.207 297.150,342.594 298.800 C 341.982 300.450,340.988 302.790,340.386 304.000 C 339.783 305.210,338.809 307.190,338.219 308.400 C 336.521 311.888,334.006 316.525,333.453 317.191 C 333.174 317.526,331.621 319.960,330.002 322.600 C 328.382 325.240,326.632 327.940,326.112 328.600 C 325.592 329.260,324.859 330.209,324.483 330.709 C 324.107 331.210,323.484 332.020,323.098 332.509 C 322.711 332.999,321.399 334.714,320.181 336.319 C 316.661 340.962,306.525 351.496,301.607 355.624 C 300.181 356.821,298.458 358.340,297.779 359.000 C 297.099 359.660,295.520 360.920,294.270 361.800 C 293.019 362.680,291.322 363.899,290.498 364.510 C 282.970 370.086,271.958 376.559,264.800 379.614 C 260.580 381.415,257.567 382.766,256.400 383.381 C 255.850 383.671,254.770 384.036,254.000 384.193 C 253.230 384.350,252.237 384.707,251.794 384.987 C 251.351 385.266,250.361 385.624,249.594 385.782 C 248.827 385.941,247.480 386.394,246.600 386.789 C 244.745 387.623,242.527 388.248,238.600 389.043 C 237.060 389.355,234.900 389.868,233.800 390.185 C 215.482 395.454,177.421 394.618,159.800 388.559 C 158.260 388.029,156.594 387.597,156.099 387.598 C 155.603 387.599,154.525 387.240,153.704 386.800 C 152.884 386.360,151.909 386.000,151.538 386.000 C 151.168 386.000,148.375 384.988,145.332 383.752 C 142.290 382.515,138.965 381.173,137.945 380.770 C 134.901 379.568,125.094 374.420,121.600 372.191 C 116.284 368.800,108.391 363.448,107.332 362.516 C 106.635 361.902,105.465 360.920,104.732 360.333 C 99.775 356.362,98.357 355.151,96.023 352.893 C 94.581 351.497,92.433 349.420,91.250 348.277 C 90.068 347.135,87.728 344.583,86.050 342.606 C 84.373 340.630,82.730 338.725,82.400 338.374 C 80.824 336.697,78.098 333.396,77.195 332.073 C 76.642 331.263,74.752 328.542,72.995 326.028 C 71.238 323.513,69.530 320.971,69.200 320.380 C 68.038 318.298,66.794 316.183,65.800 314.600 C 65.247 313.720,64.508 312.370,64.158 311.600 C 63.807 310.830,63.297 309.930,63.024 309.600 C 62.751 309.270,61.662 307.020,60.605 304.600 C 59.548 302.180,58.170 299.083,57.542 297.717 C 56.914 296.351,56.400 294.986,56.400 294.683 C 56.400 294.381,56.267 294.001,56.105 293.838 C 55.813 293.547,54.406 289.695,53.461 286.600 C 53.192 285.720,52.664 284.424,52.286 283.720 C 51.909 283.016,51.600 282.077,51.600 281.634 C 51.600 281.191,51.307 279.967,50.948 278.914 C 50.589 277.861,49.974 275.560,49.581 273.800 C 45.115 253.804,44.864 249.086,45.175 191.000 L 45.400 149.000 46.578 146.673 C 47.651 144.553,48.290 143.834,51.243 141.417 C 52.688 140.234,54.999 140.061,69.801 140.029 C 87.130 139.992,89.052 139.565,92.557 134.974 C 95.226 131.479,95.142 133.246,95.391 75.666 C 95.641 17.885,95.414 22.297,98.405 16.949 C 103.347 8.110,116.119 3.656,124.915 7.704 " stroke="none" fill="#fff" fill-rule="evenodd"></path><path id="path2" d="M98.800 220.675 C 98.250 220.786,97.620 221.016,97.400 221.186 C 97.180 221.356,96.280 221.726,95.400 222.006 C 89.650 223.842,84.126 229.790,80.932 237.586 C 79.780 240.399,79.887 248.923,81.117 252.334 C 88.814 273.671,117.246 275.189,126.899 254.777 C 130.733 246.669,129.207 236.022,123.200 228.964 C 121.937 227.480,117.153 223.600,116.586 223.600 C 116.352 223.600,115.958 223.398,115.712 223.152 C 113.719 221.159,103.680 219.689,98.800 220.675 M286.628 223.155 C 286.507 223.350,285.732 223.631,284.904 223.780 C 269.154 226.610,261.767 249.024,272.219 262.273 C 278.217 269.877,291.346 274.360,297.929 271.053 C 298.518 270.757,299.600 270.400,300.332 270.261 C 301.065 270.121,302.125 269.645,302.686 269.204 C 303.248 268.762,303.883 268.400,304.096 268.400 C 304.310 268.400,305.050 267.942,305.742 267.383 C 306.434 266.824,307.399 266.059,307.886 265.683 C 322.250 254.614,315.927 226.991,298.296 223.783 C 297.468 223.632,296.693 223.350,296.572 223.155 C 296.451 222.960,294.214 222.800,291.600 222.800 C 288.986 222.800,286.749 222.960,286.628 223.155 M120.800 284.684 C 119.830 285.041,118.062 286.984,117.599 288.202 C 116.577 290.890,117.736 293.460,121.557 296.981 C 122.524 297.871,124.234 299.472,125.357 300.537 C 126.481 301.602,128.030 302.950,128.800 303.532 C 130.126 304.534,136.092 309.190,137.044 309.965 C 137.288 310.164,138.257 310.748,139.198 311.263 C 141.580 312.568,142.233 312.964,142.857 313.481 C 143.156 313.728,146.460 315.497,150.200 317.410 C 153.940 319.324,157.180 321.025,157.400 321.190 C 157.620 321.355,158.340 321.618,159.000 321.776 C 159.660 321.933,160.740 322.310,161.400 322.613 C 164.311 323.950,170.733 325.835,178.200 327.545 C 187.661 329.711,205.777 331.031,212.400 330.039 C 214.050 329.791,217.740 329.321,220.600 328.993 C 223.460 328.665,226.070 328.295,226.400 328.171 C 226.730 328.047,228.890 327.522,231.200 327.005 C 235.990 325.933,240.301 324.695,241.517 324.044 C 241.974 323.800,242.727 323.600,243.192 323.600 C 243.657 323.600,244.882 323.150,245.915 322.600 C 246.948 322.050,247.980 321.600,248.208 321.600 C 248.436 321.600,249.972 320.910,251.622 320.066 C 253.273 319.222,255.100 318.411,255.683 318.265 C 256.267 318.119,256.982 317.763,257.272 317.476 C 257.562 317.188,258.340 316.676,259.000 316.338 C 260.137 315.755,263.258 314.042,265.000 313.044 C 265.854 312.555,267.223 311.700,269.800 310.044 C 272.450 308.342,280.879 301.475,284.433 298.123 C 288.095 294.670,287.697 289.881,283.566 287.676 C 280.865 286.234,278.182 287.091,274.187 290.671 C 271.969 292.658,268.449 295.558,267.200 296.426 C 266.430 296.962,264.990 298.030,264.000 298.800 C 259.980 301.927,253.925 305.414,248.000 308.015 C 245.910 308.932,243.750 309.911,243.200 310.190 C 242.650 310.469,241.508 310.920,240.661 311.192 C 239.815 311.465,238.645 311.937,238.061 312.241 C 237.478 312.546,236.673 312.796,236.274 312.798 C 235.875 312.799,234.705 313.139,233.674 313.552 C 231.158 314.563,223.705 316.330,220.000 316.794 C 218.350 317.002,215.740 317.352,214.200 317.572 C 210.801 318.060,198.681 318.074,194.000 317.596 C 187.849 316.968,181.820 316.068,180.600 315.596 C 179.940 315.341,178.230 314.890,176.800 314.594 C 174.164 314.048,171.183 313.121,169.000 312.170 C 168.340 311.882,166.540 311.180,165.000 310.610 C 157.435 307.809,148.744 303.093,142.979 298.662 C 141.890 297.825,139.752 296.259,138.227 295.182 C 136.702 294.105,133.912 291.725,132.027 289.893 C 126.249 284.279,124.326 283.386,120.800 284.684 " stroke="none" fill="#fff" fill-rule="evenodd"></path><path id="path3" d="M111.000 6.976 C 104.355 8.868,98.789 14.406,96.436 21.466 C 95.629 23.887,95.611 24.853,95.391 75.666 C 95.142 133.246,95.226 131.479,92.557 134.974 C 89.052 139.565,87.130 139.992,69.801 140.029 C 54.999 140.061,52.688 140.234,51.243 141.417 C 48.290 143.834,47.651 144.553,46.578 146.673 L 45.400 149.000 45.175 191.000 C 44.888 244.696,45.250 253.704,48.267 267.800 C 49.677 274.389,50.335 277.114,50.948 278.914 C 51.307 279.967,51.600 281.191,51.600 281.634 C 51.600 282.077,51.909 283.016,52.286 283.720 C 52.664 284.424,53.192 285.720,53.461 286.600 C 54.406 289.695,55.813 293.547,56.105 293.838 C 56.267 294.001,56.400 294.381,56.400 294.683 C 56.400 294.986,56.914 296.351,57.542 297.717 C 58.170 299.083,59.548 302.180,60.605 304.600 C 61.662 307.020,62.751 309.270,63.024 309.600 C 63.297 309.930,63.807 310.830,64.158 311.600 C 64.508 312.370,65.247 313.720,65.800 314.600 C 66.794 316.183,68.038 318.298,69.200 320.380 C 69.530 320.971,71.238 323.513,72.995 326.028 C 74.752 328.542,76.642 331.263,77.195 332.073 C 78.098 333.396,80.824 336.697,82.400 338.374 C 82.730 338.725,84.373 340.630,86.050 342.606 C 87.728 344.583,90.068 347.135,91.250 348.277 C 92.433 349.420,94.581 351.497,96.023 352.893 C 98.357 355.151,99.775 356.362,104.732 360.333 C 105.465 360.920,106.635 361.902,107.332 362.516 C 108.391 363.448,116.284 368.800,121.600 372.191 C 125.094 374.420,134.901 379.568,137.945 380.770 C 138.965 381.173,142.290 382.515,145.332 383.752 C 148.375 384.988,151.168 386.000,151.538 386.000 C 151.909 386.000,152.884 386.360,153.704 386.800 C 154.525 387.240,155.603 387.599,156.099 387.598 C 156.594 387.597,158.260 388.029,159.800 388.559 C 177.421 394.618,215.482 395.454,233.800 390.185 C 234.900 389.868,237.060 389.355,238.600 389.043 C 242.527 388.248,244.745 387.623,246.600 386.789 C 247.480 386.394,248.827 385.941,249.594 385.782 C 250.361 385.624,251.351 385.266,251.794 384.987 C 252.237 384.707,253.230 384.350,254.000 384.193 C 254.770 384.036,255.850 383.671,256.400 383.381 C 257.567 382.766,260.580 381.415,264.800 379.614 C 271.958 376.559,282.970 370.086,290.498 364.510 C 291.322 363.899,293.019 362.680,294.270 361.800 C 295.520 360.920,297.099 359.660,297.779 359.000 C 298.458 358.340,300.181 356.821,301.607 355.624 C 306.525 351.496,316.661 340.962,320.181 336.319 C 321.399 334.714,322.711 332.999,323.098 332.509 C 323.484 332.020,324.107 331.210,324.483 330.709 C 324.859 330.209,325.592 329.260,326.112 328.600 C 326.632 327.940,328.382 325.240,330.002 322.600 C 331.621 319.960,333.174 317.526,333.453 317.191 C 334.006 316.525,336.521 311.888,338.219 308.400 C 338.809 307.190,339.783 305.210,340.386 304.000 C 340.988 302.790,341.982 300.450,342.594 298.800 C 343.207 297.150,344.044 295.037,344.454 294.104 C 344.864 293.172,345.200 292.209,345.200 291.965 C 345.200 291.721,345.457 290.954,345.772 290.260 C 347.085 287.368,348.400 283.681,348.400 282.893 C 348.400 282.429,348.827 280.643,349.348 278.925 C 351.218 272.766,351.393 271.921,353.348 259.600 C 355.039 248.948,356.035 153.777,354.508 148.740 C 353.511 145.449,352.126 143.592,349.245 141.679 L 347.016 140.200 331.208 139.947 C 311.853 139.638,310.973 139.459,307.557 135.146 C 304.883 131.769,305.018 134.816,305.009 77.800 C 305.003 41.371,304.867 25.431,304.549 23.584 C 303.955 20.135,300.694 13.842,298.823 12.531 C 298.287 12.156,297.536 11.568,297.153 11.224 C 288.019 3.027,273.352 5.377,266.197 16.183 C 262.591 21.630,262.537 22.731,262.445 93.687 L 262.400 128.374 261.500 131.269 C 260.204 135.439,257.669 137.992,253.595 139.229 L 250.660 140.120 199.630 139.869 C 142.104 139.587,143.359 139.668,141.077 136.094 C 140.899 135.815,140.443 135.275,140.064 134.894 C 139.405 134.229,139.110 133.674,138.002 131.000 C 137.452 129.673,137.402 108.391,137.829 57.200 C 138.108 23.782,138.058 23.082,134.954 17.196 C 130.449 8.652,120.308 4.325,111.000 6.976 M121.851 8.769 C 122.438 9.231,123.497 9.600,124.233 9.600 C 125.093 9.600,125.938 9.952,126.638 10.600 C 127.231 11.150,128.051 11.600,128.458 11.600 C 128.866 11.600,129.200 11.876,129.200 12.213 C 129.200 12.550,129.570 12.963,130.022 13.130 C 130.856 13.438,131.457 14.284,132.632 16.800 C 132.992 17.570,133.526 18.338,133.820 18.507 C 134.113 18.675,134.476 19.665,134.625 20.707 C 134.775 21.748,135.191 22.934,135.549 23.342 C 136.142 24.018,136.209 28.921,136.304 78.816 C 136.407 133.055,136.416 133.551,137.204 133.801 C 137.642 133.940,138.001 134.357,138.002 134.727 C 138.003 135.097,138.543 135.975,139.202 136.678 C 139.861 137.381,140.400 138.102,140.400 138.281 C 140.400 138.459,140.985 138.855,141.700 139.161 C 142.415 139.466,143.334 140.005,143.742 140.358 C 144.939 141.393,255.819 141.336,256.576 140.300 C 256.858 139.915,257.248 139.595,257.444 139.588 C 258.028 139.569,260.400 137.253,260.400 136.702 C 260.400 136.422,260.760 135.968,261.200 135.693 C 261.640 135.418,262.000 134.835,262.000 134.397 C 262.000 133.958,262.384 133.216,262.854 132.746 L 263.707 131.893 263.754 78.764 C 263.797 28.697,263.840 25.605,264.499 25.117 C 264.971 24.767,265.281 23.791,265.456 22.100 C 265.628 20.429,265.889 19.600,266.241 19.600 C 266.531 19.600,266.984 18.880,267.247 18.000 C 267.511 17.120,267.878 16.400,268.062 16.400 C 268.247 16.400,268.623 15.877,268.899 15.238 C 269.421 14.027,272.812 11.200,273.741 11.200 C 274.035 11.200,274.602 10.840,275.000 10.400 C 275.398 9.960,276.224 9.600,276.836 9.600 C 278.200 9.600,280.081 8.916,280.478 8.275 C 280.868 7.645,287.947 7.668,288.338 8.300 C 288.693 8.873,290.589 9.600,291.730 9.600 C 292.213 9.600,292.818 9.937,293.076 10.350 C 293.333 10.762,294.043 11.225,294.654 11.378 C 295.265 11.531,296.007 12.004,296.305 12.428 C 296.602 12.853,297.066 13.200,297.335 13.200 C 297.604 13.200,298.051 13.695,298.327 14.300 C 299.229 16.274,300.316 18.050,300.916 18.530 C 301.240 18.788,301.625 19.805,301.772 20.790 C 301.918 21.774,302.345 22.918,302.719 23.331 C 303.352 24.031,303.408 27.902,303.510 78.389 C 303.618 131.893,303.631 132.704,304.410 133.409 C 304.844 133.802,305.200 134.365,305.200 134.659 C 305.200 134.953,305.560 135.418,306.000 135.693 C 306.440 135.968,306.800 136.400,306.800 136.655 C 306.800 137.214,309.449 139.600,310.070 139.600 C 310.313 139.600,310.742 139.915,311.023 140.300 C 311.493 140.943,312.963 141.016,328.955 141.200 C 344.792 141.382,346.449 141.464,347.192 142.100 C 347.641 142.485,348.367 142.800,348.805 142.800 C 349.345 142.800,349.600 143.068,349.600 143.637 C 349.600 144.125,349.919 144.558,350.362 144.674 C 350.849 144.802,351.218 145.369,351.382 146.246 C 351.524 147.000,351.991 147.837,352.420 148.105 C 353.522 148.793,353.662 245.723,352.563 247.400 C 352.079 248.138,351.849 250.075,351.601 255.486 C 351.357 260.822,351.109 262.923,350.612 263.892 C 350.205 264.685,349.865 266.617,349.733 268.892 C 349.600 271.161,349.238 273.221,348.801 274.200 C 348.407 275.080,347.976 277.240,347.842 279.000 C 347.676 281.187,347.405 282.344,346.986 282.654 C 346.640 282.910,346.264 284.042,346.123 285.254 C 345.957 286.671,345.622 287.578,345.136 287.923 C 344.663 288.259,344.400 288.942,344.400 289.835 C 344.400 290.671,344.172 291.313,343.828 291.445 C 343.177 291.695,342.400 293.661,342.400 295.060 C 342.400 295.577,342.040 296.360,341.600 296.800 C 341.160 297.240,340.800 298.022,340.800 298.539 C 340.800 299.055,340.451 299.880,340.025 300.372 C 339.598 300.863,339.134 301.790,338.993 302.431 C 338.853 303.071,338.476 303.742,338.156 303.921 C 337.836 304.100,337.460 304.950,337.321 305.811 C 337.181 306.672,336.758 307.708,336.380 308.113 C 336.003 308.518,335.581 309.561,335.442 310.430 C 335.302 311.300,334.831 312.369,334.394 312.806 C 333.957 313.243,333.600 313.948,333.600 314.373 C 333.600 314.798,333.240 315.260,332.800 315.400 C 332.360 315.540,332.000 315.912,332.000 316.227 C 332.000 316.542,331.658 317.142,331.239 317.561 C 330.821 317.979,330.345 318.744,330.182 319.261 C 330.019 319.777,329.551 320.435,329.143 320.723 C 328.734 321.010,328.400 321.589,328.400 322.009 C 328.400 322.428,328.052 322.958,327.627 323.186 C 327.202 323.413,326.730 323.991,326.578 324.469 C 326.426 324.947,325.964 325.701,325.551 326.145 C 325.138 326.588,324.800 327.181,324.800 327.461 C 324.800 327.742,324.440 328.165,324.000 328.400 C 323.560 328.635,323.200 329.078,323.200 329.383 C 323.200 329.689,322.750 330.473,322.200 331.127 C 321.650 331.781,321.200 332.584,321.200 332.911 C 321.200 333.239,320.840 333.601,320.400 333.716 C 319.960 333.831,319.600 334.191,319.600 334.517 C 319.600 334.842,319.150 335.403,318.600 335.763 C 318.050 336.124,317.600 336.684,317.600 337.009 C 317.600 337.334,317.425 337.600,317.211 337.600 C 316.420 337.600,310.425 344.388,310.411 345.300 C 310.405 345.685,310.126 346.000,309.790 346.000 C 309.455 346.000,309.009 346.450,308.800 347.000 C 308.591 347.550,308.157 348.000,307.837 348.000 C 307.516 348.000,307.140 348.360,307.000 348.800 C 306.860 349.240,306.408 349.600,305.996 349.600 C 305.583 349.600,305.010 349.918,304.723 350.307 C 304.435 350.696,303.750 351.310,303.200 351.673 C 301.671 352.680,296.978 357.310,296.685 358.100 C 296.542 358.485,296.156 358.800,295.827 358.800 C 295.497 358.800,295.035 359.160,294.800 359.600 C 294.565 360.040,294.030 360.400,293.613 360.400 C 293.195 360.400,292.746 360.741,292.613 361.158 C 292.465 361.626,291.847 362.015,290.997 362.174 C 290.241 362.316,289.482 362.664,289.311 362.947 C 289.140 363.231,288.415 363.802,287.700 364.218 C 286.985 364.633,286.400 365.114,286.400 365.286 C 286.400 365.459,286.162 365.600,285.872 365.600 C 285.582 365.600,284.953 366.050,284.475 366.600 C 283.997 367.150,283.355 367.600,283.049 367.600 C 282.744 367.600,282.399 367.960,282.284 368.400 C 282.143 368.941,281.738 369.200,281.037 369.200 C 280.467 369.200,279.640 369.560,279.200 370.000 C 278.760 370.440,278.246 370.800,278.058 370.800 C 277.871 370.800,277.231 371.250,276.638 371.800 C 276.044 372.350,275.210 372.800,274.783 372.800 C 274.356 372.800,273.782 373.160,273.507 373.600 C 273.232 374.040,272.699 374.400,272.323 374.400 C 271.431 374.400,269.600 375.238,269.600 375.646 C 269.600 375.821,268.925 376.079,268.100 376.219 C 267.275 376.360,266.150 376.790,265.600 377.175 C 265.050 377.560,263.942 377.998,263.139 378.148 C 262.335 378.299,261.579 378.677,261.459 378.990 C 261.339 379.302,260.530 379.749,259.662 379.983 C 258.793 380.217,257.979 380.677,257.854 381.004 C 257.704 381.394,257.074 381.600,256.035 381.600 C 254.910 381.600,254.285 381.829,253.897 382.383 C 253.564 382.859,252.651 383.274,251.574 383.439 C 250.598 383.589,249.478 384.046,249.084 384.456 C 248.625 384.934,247.807 385.200,246.798 385.200 C 245.677 385.200,245.114 385.413,244.830 385.944 C 244.504 386.553,243.800 386.725,240.916 386.899 C 238.982 387.015,236.590 387.413,235.600 387.784 C 234.610 388.154,232.306 388.609,230.479 388.796 C 228.652 388.983,226.754 389.400,226.260 389.724 C 225.633 390.134,223.609 390.395,219.581 390.584 C 214.999 390.799,213.648 390.995,213.059 391.528 C 211.921 392.558,190.621 392.537,188.099 391.503 C 186.878 391.002,184.751 390.739,180.567 390.571 C 175.998 390.386,174.462 390.180,173.468 389.614 C 172.588 389.114,170.975 388.822,168.200 388.663 C 165.306 388.496,163.762 388.207,162.615 387.616 C 161.557 387.071,160.262 386.800,158.715 386.800 C 157.011 386.800,156.400 386.654,156.400 386.247 C 156.400 385.601,155.192 385.213,153.154 385.205 C 152.355 385.202,151.281 384.865,150.754 384.451 C 150.229 384.039,149.035 383.573,148.100 383.416 C 147.115 383.250,146.400 382.910,146.400 382.608 C 146.400 381.980,145.519 381.600,144.064 381.600 C 143.426 381.600,142.790 381.276,142.493 380.800 C 142.174 380.289,141.560 380.000,140.793 380.000 C 140.082 380.000,139.309 379.669,138.898 379.189 C 138.516 378.743,137.536 378.271,136.720 378.141 C 135.904 378.010,134.931 377.565,134.556 377.152 C 134.182 376.738,133.552 376.400,133.156 376.400 C 132.760 376.400,131.772 375.950,130.962 375.400 C 130.151 374.850,129.017 374.400,128.440 374.400 C 127.826 374.400,127.186 374.069,126.893 373.600 C 126.618 373.160,126.044 372.800,125.617 372.800 C 125.190 372.800,124.356 372.350,123.762 371.800 C 123.169 371.250,122.479 370.800,122.230 370.800 C 121.982 370.800,121.470 370.440,121.093 370.000 C 120.716 369.560,120.164 369.200,119.866 369.200 C 119.568 369.200,119.013 368.856,118.633 368.436 C 118.252 368.016,117.526 367.568,117.018 367.440 C 116.511 367.313,116.001 366.847,115.885 366.404 C 115.770 365.962,115.397 365.600,115.058 365.600 C 114.719 365.600,113.956 365.150,113.362 364.600 C 112.769 364.050,112.129 363.600,111.942 363.600 C 111.754 363.600,111.285 363.325,110.900 362.990 C 108.332 360.752,107.767 360.400,106.742 360.400 C 105.870 360.400,105.600 360.212,105.600 359.608 C 105.600 359.096,105.246 358.726,104.600 358.564 C 103.974 358.407,103.600 358.030,103.600 357.557 C 103.600 357.067,103.318 356.800,102.800 356.800 C 102.360 356.800,102.000 356.593,102.000 356.340 C 102.000 355.772,99.396 353.200,98.820 353.200 C 98.589 353.200,98.400 352.926,98.400 352.590 C 98.400 352.255,97.950 351.809,97.400 351.600 C 96.850 351.391,96.400 350.957,96.400 350.637 C 96.400 350.316,96.034 349.938,95.587 349.796 C 95.140 349.654,94.304 349.102,93.729 348.569 C 93.154 348.036,92.439 347.600,92.142 347.600 C 91.844 347.600,91.600 347.399,91.600 347.153 C 91.600 346.907,91.060 346.091,90.400 345.339 C 89.740 344.587,89.200 343.708,89.200 343.386 C 89.200 343.064,88.926 342.800,88.590 342.800 C 88.255 342.800,87.809 342.350,87.600 341.800 C 87.391 341.250,87.031 340.800,86.800 340.800 C 86.569 340.800,86.209 340.350,86.000 339.800 C 85.791 339.250,85.345 338.800,85.010 338.800 C 84.674 338.800,84.400 338.526,84.400 338.190 C 84.400 337.855,83.950 337.409,83.400 337.200 C 82.850 336.991,82.400 336.557,82.400 336.237 C 82.400 335.916,82.040 335.540,81.600 335.400 C 81.160 335.260,80.800 334.871,80.800 334.536 C 80.800 334.200,80.457 333.835,80.038 333.726 C 79.546 333.597,79.182 333.027,79.011 332.119 C 78.839 331.202,78.478 330.641,77.974 330.509 C 77.497 330.384,77.200 329.960,77.200 329.402 C 77.200 328.905,76.902 328.251,76.538 327.949 C 76.174 327.647,75.632 326.815,75.333 326.100 C 75.034 325.385,74.522 324.800,74.195 324.800 C 73.868 324.800,73.600 324.513,73.600 324.163 C 73.600 323.520,72.553 321.964,70.900 320.151 C 70.405 319.608,70.000 318.838,70.000 318.440 C 70.000 318.042,69.550 317.231,69.000 316.638 C 68.450 316.044,68.000 315.304,68.000 314.993 C 68.000 314.683,67.640 314.235,67.200 314.000 C 66.760 313.765,66.400 313.218,66.400 312.786 C 66.400 312.354,66.040 311.640,65.600 311.200 C 65.160 310.760,64.799 310.085,64.799 309.700 C 64.798 309.315,64.448 308.428,64.022 307.729 C 63.596 307.030,63.129 305.827,62.984 305.055 C 62.839 304.284,62.379 303.343,61.960 302.964 C 61.542 302.586,61.200 301.773,61.200 301.159 C 61.200 300.521,60.771 299.578,60.200 298.962 C 59.631 298.348,59.200 297.403,59.200 296.769 C 59.200 296.030,58.947 295.574,58.451 295.416 C 57.910 295.245,57.645 294.674,57.497 293.367 C 57.385 292.370,57.004 291.342,56.649 291.078 C 56.292 290.813,55.886 289.709,55.738 288.600 C 55.571 287.345,55.197 286.405,54.736 286.077 C 54.315 285.778,54.000 285.065,54.000 284.410 C 54.000 282.904,53.631 282.000,53.017 282.000 C 52.676 282.000,52.438 280.849,52.259 278.329 C 52.114 276.277,51.750 274.278,51.435 273.798 C 51.125 273.325,50.690 271.118,50.467 268.895 C 50.244 266.671,49.794 264.255,49.467 263.526 C 49.104 262.715,48.794 260.387,48.671 257.533 C 48.525 254.159,48.266 252.442,47.735 251.333 C 46.619 249.006,46.586 150.142,47.700 148.911 C 48.085 148.485,48.400 147.738,48.400 147.250 C 48.400 146.763,48.850 145.914,49.400 145.364 C 49.950 144.814,50.400 144.035,50.400 143.633 C 50.400 143.093,50.807 142.837,51.956 142.653 C 52.812 142.517,53.614 142.141,53.738 141.818 C 53.929 141.318,56.527 141.215,71.410 141.116 C 87.539 141.009,88.886 140.947,89.248 140.300 C 89.464 139.915,89.856 139.597,90.120 139.594 C 90.864 139.584,93.200 137.369,93.200 136.674 C 93.200 136.333,93.539 135.946,93.952 135.815 C 94.387 135.677,94.810 135.051,94.954 134.331 C 95.091 133.646,95.517 132.856,95.901 132.575 C 96.559 132.094,96.606 128.939,96.704 78.432 C 96.762 48.935,96.958 24.800,97.140 24.800 C 97.673 24.800,98.400 22.873,98.400 21.459 C 98.400 20.697,98.815 19.593,99.400 18.800 C 99.950 18.055,100.400 17.229,100.400 16.965 C 100.400 16.701,100.654 16.388,100.965 16.269 C 101.275 16.150,101.761 15.410,102.045 14.626 C 102.329 13.842,102.724 13.200,102.923 13.200 C 103.122 13.200,103.819 12.750,104.473 12.200 C 105.127 11.650,105.974 11.200,106.355 11.200 C 106.737 11.200,107.436 10.840,107.908 10.400 C 108.380 9.960,109.179 9.599,109.683 9.599 C 110.187 9.598,111.164 9.253,111.854 8.832 C 114.313 7.333,119.980 7.297,121.851 8.769 " stroke="none" fill="#6fa0fc" fill-rule="evenodd"></path><path id="path4" d="M103.200 218.661 C 102.870 218.770,102.240 219.005,101.800 219.183 C 101.360 219.362,99.596 219.688,97.881 219.909 C 96.074 220.142,94.462 220.582,94.049 220.955 C 93.658 221.310,93.097 221.600,92.804 221.600 C 90.179 221.600,81.200 230.539,81.200 233.151 C 81.200 233.416,80.844 234.120,80.410 234.716 C 79.876 235.447,79.481 236.972,79.196 239.400 C 78.963 241.380,78.671 243.654,78.548 244.453 C 78.416 245.312,78.504 246.017,78.762 246.177 C 79.008 246.329,79.201 247.620,79.202 249.124 C 79.205 252.128,79.740 254.483,80.620 255.363 C 80.945 255.688,81.321 256.504,81.455 257.177 C 81.590 257.850,81.853 258.400,82.040 258.400 C 82.227 258.400,82.640 258.895,82.959 259.500 C 83.627 260.767,87.256 264.693,88.590 265.592 C 89.086 265.926,90.224 266.689,91.118 267.287 C 92.012 267.885,93.611 268.580,94.672 268.832 C 95.732 269.084,97.332 269.630,98.226 270.045 C 100.563 271.130,108.768 271.141,110.806 270.062 C 111.573 269.655,112.891 269.214,113.735 269.081 C 115.301 268.834,119.200 266.606,119.200 265.958 C 119.200 265.761,119.461 265.600,119.779 265.600 C 120.754 265.600,125.200 261.318,125.200 260.379 C 125.200 260.171,125.481 259.685,125.825 259.300 C 127.231 257.725,128.411 255.474,128.663 253.886 C 128.810 252.959,129.275 251.732,129.696 251.159 C 130.773 249.697,130.618 239.342,129.500 238.109 C 129.115 237.684,128.800 236.850,128.800 236.255 C 128.800 232.410,120.874 223.206,116.338 221.784 C 115.822 221.622,114.990 221.170,114.488 220.778 C 113.904 220.322,112.483 219.964,110.533 219.781 C 108.860 219.624,107.020 219.250,106.445 218.950 C 105.387 218.398,104.293 218.301,103.200 218.661 M112.932 221.692 C 114.214 222.248,115.465 222.905,115.712 223.152 C 115.958 223.398,116.352 223.600,116.586 223.600 C 117.153 223.600,121.937 227.480,123.200 228.964 C 125.570 231.749,127.493 235.493,128.290 238.879 C 133.479 260.902,110.090 277.119,90.600 265.011 C 82.565 260.020,77.559 245.819,80.932 237.586 C 84.126 229.790,89.650 223.842,95.400 222.006 C 96.280 221.726,97.180 221.356,97.400 221.186 C 98.959 219.980,109.797 220.333,112.932 221.692 M284.298 222.015 C 280.770 222.892,274.992 226.674,272.636 229.648 C 271.872 230.612,270.877 231.779,270.424 232.242 C 269.971 232.705,269.600 233.290,269.600 233.542 C 269.600 233.794,269.240 234.360,268.800 234.800 C 268.360 235.240,268.000 235.993,268.000 236.474 C 268.000 236.955,267.658 237.720,267.240 238.174 C 265.385 240.188,265.391 253.991,267.247 256.351 C 267.661 256.877,268.000 257.644,268.000 258.054 C 268.000 258.464,268.360 259.160,268.800 259.600 C 269.240 260.040,269.600 260.597,269.600 260.837 C 269.600 262.082,275.459 268.175,277.500 269.052 C 278.215 269.359,278.800 269.780,278.800 269.987 C 278.800 270.194,279.475 270.479,280.300 270.619 C 281.125 270.760,282.340 271.263,283.000 271.737 C 284.880 273.089,298.435 273.059,300.164 271.700 C 300.793 271.205,301.817 270.800,302.438 270.800 C 303.060 270.800,303.891 270.462,304.284 270.050 C 304.678 269.637,305.401 269.176,305.892 269.025 C 307.105 268.652,313.600 262.119,313.600 261.272 C 313.600 260.898,313.960 260.368,314.400 260.093 C 314.840 259.818,315.200 259.301,315.200 258.944 C 315.200 258.587,315.545 257.733,315.967 257.047 C 318.054 253.655,318.217 240.259,316.198 238.000 C 315.904 237.670,315.536 236.892,315.381 236.272 C 315.226 235.651,314.762 234.933,314.350 234.676 C 313.937 234.418,313.600 233.970,313.600 233.680 C 313.600 232.560,310.014 228.564,307.000 226.324 C 305.900 225.507,304.793 224.659,304.541 224.441 C 301.890 222.152,289.647 220.684,284.298 222.015 M296.572 223.155 C 296.693 223.350,297.468 223.632,298.296 223.783 C 315.927 226.991,322.250 254.614,307.886 265.683 C 307.399 266.059,306.434 266.824,305.742 267.383 C 305.050 267.942,304.310 268.400,304.096 268.400 C 303.883 268.400,303.248 268.762,302.686 269.204 C 302.125 269.645,301.065 270.121,300.332 270.261 C 299.600 270.400,298.518 270.757,297.929 271.053 C 291.346 274.360,278.217 269.877,272.219 262.273 C 261.767 249.024,269.154 226.610,284.904 223.780 C 285.732 223.631,286.507 223.350,286.628 223.155 C 286.749 222.960,288.986 222.800,291.600 222.800 C 294.214 222.800,296.451 222.960,296.572 223.155 M122.800 282.667 C 122.470 282.780,121.633 283.126,120.940 283.436 C 120.246 283.746,119.547 284.000,119.385 284.000 C 116.586 284.000,115.526 292.345,117.952 295.293 C 119.215 296.829,124.641 302.162,126.403 303.600 C 127.077 304.150,128.080 305.005,128.633 305.500 C 129.185 305.995,129.782 306.400,129.960 306.400 C 130.138 306.400,130.746 306.805,131.311 307.300 C 131.876 307.795,133.001 308.650,133.810 309.200 C 134.619 309.750,135.623 310.515,136.040 310.900 C 136.458 311.285,137.025 311.603,137.300 311.607 C 137.575 311.611,138.224 312.061,138.742 312.607 C 139.260 313.153,140.019 313.600,140.428 313.600 C 140.837 313.600,141.365 313.960,141.600 314.400 C 141.835 314.840,142.279 315.200,142.586 315.200 C 142.892 315.200,143.399 315.456,143.711 315.769 C 144.252 316.309,146.216 317.417,148.400 318.415 C 148.950 318.666,150.237 319.305,151.259 319.836 C 152.281 320.366,153.406 320.800,153.759 320.800 C 154.112 320.800,154.400 320.958,154.400 321.151 C 154.400 321.588,156.125 322.261,158.061 322.581 C 158.865 322.713,159.618 323.069,159.734 323.371 C 159.849 323.673,160.722 324.051,161.672 324.211 C 162.622 324.371,163.829 324.839,164.354 325.251 C 164.879 325.663,165.779 326.001,166.354 326.003 C 166.929 326.004,167.940 326.363,168.600 326.800 C 169.319 327.276,170.477 327.596,171.488 327.597 C 172.416 327.599,173.856 327.947,174.688 328.372 C 175.519 328.796,177.280 329.257,178.600 329.396 C 179.920 329.535,181.630 329.895,182.400 330.195 C 183.170 330.496,186.320 330.944,189.400 331.191 C 192.480 331.439,196.538 331.831,198.419 332.063 C 203.084 332.637,223.578 331.153,224.845 330.149 C 225.150 329.907,226.642 329.599,228.159 329.464 C 229.707 329.327,231.600 328.864,232.472 328.410 C 233.326 327.964,234.715 327.600,235.558 327.600 C 236.441 327.600,237.497 327.286,238.046 326.860 C 238.571 326.453,239.779 326.018,240.731 325.894 C 241.683 325.769,242.763 325.407,243.131 325.088 C 243.499 324.769,244.519 324.398,245.399 324.263 C 246.278 324.129,247.204 323.736,247.456 323.391 C 247.708 323.046,248.609 322.654,249.457 322.521 C 250.306 322.387,251.429 321.945,251.954 321.539 C 252.479 321.132,253.216 320.800,253.594 320.800 C 253.971 320.800,255.612 320.090,257.240 319.222 C 258.868 318.354,260.740 317.392,261.400 317.084 C 262.060 316.776,262.847 316.226,263.149 315.862 C 263.451 315.498,263.948 315.200,264.253 315.200 C 264.558 315.200,265.116 314.840,265.493 314.400 C 265.870 313.960,266.506 313.600,266.908 313.600 C 267.310 313.600,268.035 313.182,268.519 312.670 C 269.004 312.159,269.715 311.624,270.100 311.483 C 270.485 311.341,270.800 310.949,270.800 310.613 C 270.800 310.276,271.096 310.000,271.459 310.000 C 272.252 310.000,275.760 307.719,277.575 306.023 C 278.295 305.350,278.973 304.800,279.082 304.800 C 279.190 304.800,279.846 304.253,280.540 303.584 C 282.992 301.219,283.488 300.800,283.842 300.791 C 284.177 300.783,284.768 300.139,287.405 296.907 C 288.836 295.153,288.974 290.863,287.645 289.448 C 287.180 288.953,286.793 288.380,286.785 288.174 C 286.696 285.890,278.658 284.714,276.796 286.712 C 276.341 287.201,275.796 287.600,275.584 287.600 C 275.373 287.600,274.860 287.915,274.444 288.300 C 274.028 288.685,272.653 289.900,271.388 291.000 C 270.124 292.100,268.318 293.765,267.376 294.700 C 266.434 295.635,265.388 296.400,265.052 296.400 C 264.716 296.400,263.956 296.850,263.362 297.400 C 262.769 297.950,262.079 298.400,261.830 298.400 C 261.582 298.400,261.070 298.760,260.693 299.200 C 260.316 299.640,259.758 300.000,259.453 300.000 C 259.148 300.000,258.653 300.295,258.355 300.655 C 258.056 301.015,257.089 301.646,256.206 302.057 C 255.323 302.468,254.060 303.164,253.400 303.603 C 252.740 304.042,251.570 304.689,250.800 305.041 C 250.030 305.392,249.130 305.912,248.800 306.195 C 248.470 306.479,247.614 306.836,246.897 306.988 C 246.180 307.140,245.409 307.610,245.183 308.033 C 244.900 308.560,244.343 308.800,243.402 308.800 C 242.587 308.800,241.741 309.102,241.316 309.546 C 240.889 309.990,239.734 310.402,238.456 310.565 C 237.277 310.716,235.913 311.101,235.425 311.420 C 234.937 311.740,233.472 312.187,232.169 312.414 C 230.866 312.640,229.065 313.093,228.167 313.420 C 227.269 313.747,224.659 314.269,222.367 314.579 C 220.075 314.890,217.390 315.329,216.400 315.555 C 212.808 316.374,191.977 316.107,188.703 315.200 C 187.115 314.760,184.958 314.400,183.910 314.400 C 182.861 314.400,180.967 314.049,179.702 313.620 C 178.436 313.191,176.230 312.638,174.800 312.391 C 173.370 312.144,171.825 311.701,171.367 311.406 C 170.908 311.110,169.669 310.739,168.613 310.581 C 167.557 310.423,166.357 309.957,165.947 309.547 C 165.506 309.106,164.634 308.800,163.820 308.800 C 163.061 308.800,162.296 308.556,162.120 308.258 C 161.697 307.542,160.246 306.800,159.269 306.800 C 158.838 306.800,158.386 306.540,158.264 306.222 C 158.142 305.904,157.415 305.527,156.649 305.383 C 155.882 305.239,155.161 304.875,155.045 304.573 C 154.929 304.272,154.242 303.743,153.517 303.398 C 151.207 302.299,149.822 301.515,149.000 300.841 C 148.560 300.481,147.121 299.649,145.803 298.993 C 144.485 298.337,143.169 297.485,142.880 297.100 C 142.591 296.715,142.095 296.400,141.777 296.400 C 141.460 296.400,141.200 296.251,141.200 296.070 C 141.200 295.888,140.066 294.820,138.680 293.695 C 136.416 291.859,132.842 288.659,130.558 286.424 C 130.095 285.971,129.553 285.600,129.354 285.600 C 129.155 285.600,128.768 285.240,128.493 284.800 C 128.218 284.360,127.770 283.994,127.497 283.986 C 127.223 283.979,126.280 283.625,125.400 283.200 C 123.667 282.363,123.684 282.366,122.800 282.667 M127.269 285.481 C 128.001 286.076,130.142 288.061,132.027 289.893 C 133.912 291.725,136.702 294.105,138.227 295.182 C 139.752 296.259,141.890 297.825,142.979 298.662 C 148.744 303.093,157.435 307.809,165.000 310.610 C 166.540 311.180,168.340 311.882,169.000 312.170 C 171.183 313.121,174.164 314.048,176.800 314.594 C 178.230 314.890,179.940 315.341,180.600 315.596 C 181.820 316.068,187.849 316.968,194.000 317.596 C 198.681 318.074,210.801 318.060,214.200 317.572 C 215.740 317.352,218.350 317.002,220.000 316.794 C 223.705 316.330,231.158 314.563,233.674 313.552 C 234.705 313.139,235.875 312.799,236.274 312.798 C 236.673 312.796,237.478 312.546,238.061 312.241 C 238.645 311.937,239.815 311.465,240.661 311.192 C 241.508 310.920,242.650 310.469,243.200 310.190 C 243.750 309.911,245.910 308.932,248.000 308.015 C 253.925 305.414,259.980 301.927,264.000 298.800 C 264.990 298.030,266.430 296.962,267.200 296.426 C 268.449 295.558,271.969 292.658,274.187 290.671 C 279.892 285.558,284.781 285.858,286.692 291.440 C 287.764 294.573,286.496 296.581,280.000 302.035 C 272.036 308.721,269.185 310.736,262.200 314.613 C 261.100 315.223,259.660 316.000,259.000 316.338 C 258.340 316.676,257.562 317.188,257.272 317.476 C 256.982 317.763,256.267 318.119,255.683 318.265 C 255.100 318.411,253.273 319.222,251.622 320.066 C 249.972 320.910,248.436 321.600,248.208 321.600 C 247.980 321.600,246.948 322.050,245.915 322.600 C 244.882 323.150,243.657 323.600,243.192 323.600 C 242.727 323.600,241.974 323.800,241.517 324.044 C 240.301 324.695,235.990 325.933,231.200 327.005 C 228.890 327.522,226.730 328.047,226.400 328.171 C 226.070 328.295,223.460 328.665,220.600 328.993 C 217.740 329.321,214.050 329.791,212.400 330.039 C 205.777 331.031,187.661 329.711,178.200 327.545 C 170.733 325.835,164.311 323.950,161.400 322.613 C 160.740 322.310,159.660 321.933,159.000 321.776 C 158.340 321.618,157.620 321.355,157.400 321.190 C 157.180 321.025,153.940 319.324,150.200 317.410 C 146.460 315.497,143.156 313.728,142.857 313.481 C 142.233 312.964,141.580 312.568,139.198 311.263 C 138.257 310.748,137.288 310.164,137.044 309.965 C 136.092 309.190,130.126 304.534,128.800 303.532 C 128.030 302.950,126.481 301.602,125.357 300.537 C 124.234 299.472,122.524 297.871,121.557 296.981 C 116.735 292.537,115.976 289.123,119.114 285.990 C 121.143 283.963,125.098 283.716,127.269 285.481 " stroke="none" fill="#5a87e0" fill-rule="evenodd"></path></g></svg>',
				"description"          => esc_html__(
					"wpForo is well integrated with Ultimate Members plugin. The integration includes changing forum profile page URL to Ultimate Members profile page URL, forum tabs in Ultimate Members profile page. Also, wpForo is integrated with Ultimate Members' Real-time Notifications addon.",
					"wpforo"
				),
				"description_original" => "wpForo is well integrated with Ultimate Members plugin. The integration includes changing forum profile page URL to Ultimate Members profile page URL, forum tabs in Ultimate Members profile page. Also, wpForo is integrated with Ultimate Members' Real-time Notifications addon.",
				"docurl"               => "",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/um.php' );
				},
				"options"              => [
					"notification" => [
						"type"                 => "radio",
						"label"                => esc_html__( "Ultimate Member Notification Integration", "wpforo" ),
						"label_original"       => "Ultimate Member Notification Integration",
						"description"          => esc_html__(
							                          "Creates notification on new forum reply in Ultimate Member Real-time Notification system. This integration requires",
							                          "wpforo"
						                          ) . ' <a href="https://ultimatemember.com/extensions/real-time-notifications/" target="_blank">Real-time Notifications</a> ' . __(
							                          'plugin.',
							                          'wpforo'
						                          ),
						"description_original" => "Creates notification on new forum reply in Ultimate Member Real-time Notification system.  This integration requires" . ' <a href="https://ultimatemember.com/extensions/real-time-notifications/" target="_blank">Real-time Notifications</a> ',
						"docurl"               => "",
					],
					"forum_tab"    => [
						"type"                 => "radio",
						"label"                => esc_html__( "Ultimate Member Profile Forum Tab Integration", "wpforo" ),
						"label_original"       => "Ultimate Member Profile Forum Tab Integration",
						"description"          => esc_html__(
							"Adds 'Forums' tab with 'Created Topics', 'Posted Replies', 'Liked Posts' and 'Subscriptions' sub-tabs to Ultimate Member Profile page.",
							"wpforo"
						),
						"description_original" => "Adds 'Forums' tab with 'Created Topics', 'Posted Replies', 'Liked Posts' and 'Subscriptions' sub-tabs to Ultimate Member Profile page.",
						"docurl"               => "",
					],
				],
			],
			'legal'         => [
				"title"                => esc_html__( "Privacy & Forum Rules", "wpforo" ),
				"title_original"       => "Privacy & Rules",
				"icon"                 => '<svg xmlns="http://www.w3.org/2000/svg" height="60px" shape-rendering="geometricPrecision" text-rendering="geometricPrecision" image-rendering="optimizeQuality" fill-rule="evenodd" clip-rule="evenodd" viewBox="0 0 498 512.17"><path d="M232.21 0c86.9 55.08 165.4 81.14 232.78 74.98 3.67 74.22-2.36 138.96-17.12 194.7-10-4.08-20.68-6.52-31.7-7 11.57-46.07 16.23-99.25 13.23-159.92-57.04 5.22-123.5-16.84-197.06-63.48C168.68 88.73 103 103.21 36.04 99.7c-2.97 113.09 16.9 198.24 55.29 260.18 28.38-23.73 76.71-20.15 99.6-51.62 1.65-2.43 2.41-3.74 2.39-4.81-.01-.56-24.83-31-27.06-34.55-5.85-9.3-16.8-21.93-16.8-32.82 0-6.15 4.85-14.17 11.8-15.96-.54-9.22-.91-18.57-.91-27.84 0-5.47.1-11.01.3-16.43.3-3.44.94-4.95 1.85-8.27a58.537 58.537 0 0 1 26.13-33.18c4.43-2.8 9.25-4.98 14.19-6.77 8.96-3.27 4.62-17.43 14.46-17.65 22.99-.59 60.81 19.51 75.54 35.48 9.39 10.38 14.75 21.92 15.07 35.92l-.93 40.27c4.08 1 8.66 4.19 9.66 8.28 3.15 12.71-10.04 28.53-16.18 38.64-5.65 9.33-27.26 34.79-27.28 35-.1 1.09.46 2.47 1.94 4.69 10.53 14.48 26.44 21.54 43.3 27.25-1.87 6.71-3.07 13.64-3.53 20.74-1.76 1.23-3.4 2.6-4.91 4.11l-.1.1c-6.73 6.75-10.93 16.04-10.93 26.26v93.19c-20.32 12.65-42.28 23.4-65.81 32.26C82.71 457.27-6.26 322.77.34 71.37 79.43 75.51 157.03 58.41 232.21 0zm105.67 375.54h3.88v-11.95c0-19.96 7.87-38.16 20.55-51.39 12.79-13.33 30.44-21.6 49.88-21.6s37.11 8.27 49.88 21.6c12.69 13.23 20.56 31.42 20.56 51.39v11.95h3.88c6.32 0 11.49 5.18 11.49 11.5v113.63c0 6.33-5.17 11.5-11.49 11.5H337.88c-6.33 0-11.49-5.17-11.49-11.5V387.04c-.01-6.32 5.16-11.5 11.49-11.5zm65.2 73.48-11.96 31.34h42.13l-11.08-31.77c7.04-3.62 11.85-10.95 11.85-19.41 0-12.06-9.77-21.82-21.84-21.82-12.05 0-21.82 9.76-21.82 21.82 0 8.8 5.21 16.38 12.72 19.84zm-39.57-73.48h97.35v-11.95c0-14.2-5.53-27.06-14.43-36.34-8.81-9.19-20.93-14.9-34.24-14.9-13.31 0-25.44 5.71-34.24 14.9-8.91 9.28-14.44 22.14-14.44 36.34v11.95z"/></svg>',
				"description"          => esc_html__( "", "wpforo" ),
				"description_original" => "",
				"docurl"               => "",
				"status"               => "ok",
				"base"                 => true,
				"callback_for_page"    => function() {
					require_once( WPFORO_DIR . '/admin/settings/legal.php' );
				},
				"options"              => [
					"contact_page_url"        => [
						"label"                => esc_html__( "Contact Information", "wpforo" ),
						"label_original"       => "Contact Information",
						"description"          => esc_html__(
							                          "According to the GDPR, all users should have an option to contact website administrator in following cases: ",
							                          "wpforo"
						                          ) . '<ul class="wpf-info" style="list-style: disc; padding: 5px 0 0 0; margin: 0 20px; line-height: 14px;"><li>' . __(
							                          'Obtain personal data and created content',
							                          'wpforo'
						                          ) . '</li><li>' . __( 'Delete account with created content', 'wpforo' ) . '</li><li>' . __(
							                          'Report user data access and control issue',
							                          'wpforo'
						                          ) . '</li><li>' . __( 'Report user rights violation', 'wpforo' ) . '</li></ul>',
						"description_original" => "According to the GDPR, all users should have an option to contact website administrator in following cases: " . '<ul class="wpf-info" style="list-style: disc; padding: 5px 0 0 0; margin: 0 20px; line-height: 14px;"><li>Obtain personal data and created content</li><li>Delete account with created content</li><li>Report user data access and control issue</li><li>Report user rights violation</li></ul>',
						"type"                 => "text",
						"placeholder"          => __( "URL to contact forum administrator", "wpforo" ),
						"docurl"               => "",
					],
					"checkbox_terms_privacy"  => [
						"label"                => esc_html__( "Checkbox: I Accept Website Terms and Privacy Policy", "wpforo" ),
						"label_original"       => "Checkbox: I Accept Website Terms and Privacy Policy",
						"description"          => esc_html__(
							"If this option is enabled, users must accept forum Terms and Privacy Policy by checking the required checkbox on registration form to be able create a forum account. The checkbox label can be managed in Forums > Phrases admin page.",
							"wpforo"
						),
						"description_original" => "If this option is enabled, users must accept forum Terms and Privacy Policy by checking the required checkbox on registration form to be able create a forum account. The checkbox label can be managed in Forums > Phrases admin page.",
						"type"                 => "radio",
						"docurl"               => "",
					],
					"checkbox_email_password" => [
						"label"                => esc_html__( "Checkbox: I Agree to Receive an Email Confirmation", "wpforo" ),
						"label_original"       => "Checkbox: I Agree to Receive an Email Confirmation",
						"description"          => esc_html__(
							"If this option is enabled, users must agree to receive an email confirmation with a link to set a password by checking the required checkbox on registration form to be able create a forum account. The checkbox label can be managed in Forums > Phrases admin page.",
							"wpforo"
						),
						"description_original" => "If this option is enabled, users must agree to receive an email confirmation with a link to set a password by checking the required checkbox on registration form to be able create a forum account. The checkbox label can be managed in Forums > Phrases admin page.",
						"type"                 => "radio",
						"docurl"               => "",
					],
					"page_terms"              => [
						"label"                => esc_html__( "Page URL to Website Terms", "wpforo" ),
						"label_original"       => "Page URL to Website Terms",
						"description"          => esc_html__(
							"Please insert URLs to your website Terms and Conditions. Links to these pages will be included in registration form checkbox label (I'm agree with website terms and privacy policy).",
							"wpforo"
						),
						"description_original" => "Please insert URLs to your website Terms and Conditions. Links to these pages will be included in registration form checkbox label (I'm agree with website terms and privacy policy).",
						"type"                 => "text",
						"placeholder"          => __( 'Page URL to website terms', 'wpforo' ),
						"docurl"               => "",
					],
					"page_privacy"            => [
						"label"                => esc_html__( "Page URL to Website Privacy Policy", "wpforo" ),
						"label_original"       => "Page URL to Website Privacy Policy",
						"description"          => esc_html__(
							"Please insert URLs to your website Terms and Privacy Policy pages. Links to these pages will be included in registration form checkbox label (I'm agree with website terms and privacy policy) and in Forum Privacy Policy. The forum Privacy Policy does not cover your whole website, it is just an extension of your website main Privacy Policy. Thus it should be linked to according pages.",
							"wpforo"
						),
						"description_original" => "Please insert URLs to your website Terms and Privacy Policy pages. Links to these pages will be included in registration form checkbox label (I'm agree with website terms and privacy policy) and in Forum Privacy Policy. The forum Privacy Policy does not cover your whole website, it is just an extension of your website main Privacy Policy. Thus it should be linked to according pages.",
						"type"                 => "text",
						"placeholder"          => __( 'Page URL to website privacy policy', 'wpforo' ),
						"docurl"               => "",
					],
					"checkbox_forum_privacy"  => [
						"label"                => esc_html__( "Checkbox: I Agree to Forum Privacy Policy", "wpforo" ),
						"label_original"       => "Checkbox: I Agree to Forum Privacy Policy",
						"description"          => esc_html__(
							"If this option is enabled, users must accept forum Terms and Privacy Policy by checking the required checkbox on registration form to be able create a forum account. The checkbox label can be managed in Forums > Phrases admin page",
							"wpforo"
						),
						"description_original" => "If this option is enabled, users must accept forum Terms and Privacy Policy by checking the required checkbox on registration form to be able create a forum account. The checkbox label can be managed in Forums > Phrases admin page",
						"type"                 => "radio",
						"docurl"               => "",
					],
					"forum_privacy_text"      => [
						"type"                 => "richeditor",
						"media_buttons"        => false,
						"fullwidth"            => true,
						"label"                => esc_html__( "Forum Privacy Policy with GDPR compliant Template", "wpforo" ),
						"label_original"       => "Forum Privacy Policy with GDPR compliant Template",
						"description"          => __(
							"This is an example of forum Privacy Policy with GDPR compliant. It adapted to wpForo plugin functions and features. <u>In case you enable this privacy policy template you become responsible for the content of this template.</u> Please read this text carefully and make sure it suits your community Privacy Policy. If it doesn't, you should edit this text and adapt it to your community rules. This template includes shortcodes [forum-name] and [forum-url]. They are automatically replaced on registration page with current forum details. Don't forget to add an information about your organization, location and contacting ways (page, email, phone, etc...). Also if you have a separate privacy policy page for website please add a link to that page.",
							"wpforo"
						),
						"description_original" => "This is an example of forum Privacy Policy with GDPR compliant. It adapted to wpForo plugin functions and features. In case you enable this privacy policy template you become responsible for the content of this template. Please read this text carefully and make sure it suits your community Privacy Policy. If it doesn't, you should edit this text and adapt it to your community rules. This template includes shortcodes [forum-name] and [forum-url]. They are automatically replaced on registration page with current forum details. Don't forget to add an information about your organization, location and contacting ways (page, email, phone, etc...). Also if you have a separate privacy policy page for website please add a link to that page.",
						"docurl"               => "",
					],
					"checkbox_fb_login"       => [
						"label"                => esc_html__( "Checkbox: I Agree to create a forum account on Facebook Login", "wpforo" ),
						"label_original"       => "Checkbox: I Agree to create a forum account on Facebook Login",
						"description"          => esc_html__(
							                          "If this option is enabled, the Facebook Login button becomes not-clickable until user accept automatic account creation process based on his/her Facebook public profile information. This checkbox and appropriate information will be displayed with Facebook Login button to comply with the GDPR",
							                          "wpforo"
						                          ) . ' <a href="https://gdpr-info.eu/art-22-gdpr/" target="_blank" rel="noreferrer">(Article 22)</a> <br>' . esc_html__(
							                          "The note text and the label of this checkbox can be managed in Forums &gt; Phrases admin page. Search the label phrase, click on edit button and change it.",
							                          "wpforo"
						                          ),
						"description_original" => 'If this option is enabled, the Facebook Login button becomes not-clickable until user accept automatic account creation process based on his/her Facebook public profile information. This checkbox and appropriate information will be displayed with Facebook Login button to comply with the GDPR <a href="https://gdpr-info.eu/art-22-gdpr/" target="_blank" rel="noreferrer">(Article 22)</a> <br>The note text and the label of this checkbox can be managed in Forums &gt; Phrases admin page. Search the label phrase, click on edit button and change it.',
						"type"                 => "radio",
						"docurl"               => "",
					],
					"cookies"                 => [
						"label"                => esc_html__( "Forum Cookies", "wpforo" ),
						"label_original"       => "Forum Cookies",
						"description"          => esc_html__(
							"Please note, that this option is only related to wpForo cookies. This doesn't disable WordPress and other plugins cookies. wpForo stores a small amount of data in cookies, it used to track visited forums and topics (bold and normal titles). Also when a guest (not registered user) creates a topic or post a reply, wpForo stores guest name and email address in cookies. wpForo uses this information to detect current guest content (topics, posts) and display it to the guest even if the content is under moderation (not approved by moderators). Also wpForo stores guest name and email in cookies to keep filled these fields when he/she posts a new reply or creates a new topic.",
							"wpforo"
						),
						"description_original" => "Please note, that this option is only related to wpForo cookies. This doesn't disable WordPress and other plugins cookies. wpForo stores a small amount of data in cookies, it used to track visited forums and topics (bold and normal titles). Also when a guest (not registered user) creates a topic or post a reply, wpForo stores guest name and email address in cookies. wpForo uses this information to detect current guest content (topics, posts) and display it to the guest even if the content is under moderation (not approved by moderators). Also wpForo stores guest name and email in cookies to keep filled these fields when he/she posts a new reply or creates a new topic.",
						"type"                 => "radio",
						"docurl"               => "",
					],
					"rules_checkbox"          => [
						"label"                => esc_html__( "Checkbox: I Accept Forum Rules", "wpforo" ),
						"label_original"       => "Checkbox: I Accept Forum Rules",
						"description"          => esc_html__(
							"If this option is enabled, users must accept forum rules by checking the required checkbox on registration form to be able create a forum account. The label text of this checkbox can be managed in Forums > Phrases admin page. Search the label phrase, click on edit button and change it.",
							"wpforo"
						),
						"description_original" => "If this option is enabled, users must accept forum rules by checking the required checkbox on registration form to be able create a forum account. The label text of this checkbox can be managed in Forums > Phrases admin page. Search the label phrase, click on edit button and change it.",
						"type"                 => "radio",
						"docurl"               => "",
					],
					"rules_text"              => [
						"type"                 => "richeditor",
						"media_buttons"        => false,
						"fullwidth"            => true,
						"label"                => esc_html__( "Forum Rules Text", "wpforo" ),
						"label_original"       => "Forum Rules Text",
						"description"          => esc_html__(
							                          "This is a basic example of forum rules provided by",
							                          "wpforo"
						                          ) . '<a href="https://www.wikihow.com/Sample/Forum-Rules" target="_blank" title="Sample Forum Rules" rel="noreferrer">wikihow.com</a>' . esc_html__(
							                          'You should edit this text and adapt it to your community rules.',
							                          'wpforo'
						                          ),
						"description_original" => 'This is a basic example of forum rules provided by <a href="https://www.wikihow.com/Sample/Forum-Rules" target="_blank" title="Sample Forum Rules" rel="noreferrer">wikihow.com</a>. You should edit this text and adapt it to your community rules.',
						"docurl"               => "",
					],
				
				],
			],
		];
		
		$this->info->core = apply_filters( 'wpforo_settings_init_core_info', $this->info->core );
	}
	
	private function init_addons() {
		$this->info->addons = [];
		$this->info->addons = apply_filters( 'wpforo_settings_init_addons_info', $this->info->addons );
	}
	
	private function get_option_info( $group, $name ) {
		if( ! ( $setting = wpfval( $this->info->core, $group ) ) && ! ( $setting = wpfval( $this->info->addons, $group ) ) ) return [];
		if( ! $option = wpfval( $setting, 'options', $name ) ) return [];
		
		return (array) $option;
	}
	
	private function get_option_variants( $group, $name, $value = null ) {
		$variants = [
			[ 'value' => true, 'label' => __( 'Yes', 'wpforo' ) ],
			[ 'value' => false, 'label' => __( 'No', 'wpforo' ) ],
		];
		if( $info = $this->get_option_info( $group, $name ) ) {
			if( $_variants = (array) wpfval( $info, 'variants' ) ) $variants = $_variants;
		}
		
		return array_map( function( $variant ) use ( $value ) {
			return $this->fix_form_element_attrs( $variant, $value );
		}, $variants );
	}
	
	private function fix_form_element_attrs( $variant, $value = null ) {
		$variant            = array_merge(
			[
				'value'     => null,
				'label'     => '',
				'title'     => '',
				'id'        => '',
				'class'     => '',
				'style'     => '',
				'alt'       => '',
				'max'       => null,
				'min'       => null,
				'maxlength' => null,
				'minlength' => null,
				'autofocus' => false,
				'disabled'  => false,
				'readonly'  => false,
				'checked'   => false,
				'selected'  => false,
			],
			(array) $variant
		);
		$variant['checked'] = $variant['selected'] = in_array( $variant['value'], (array) $value, true );
		if( is_bool( $variant['value'] ) ) $variant['value'] = intval( $variant['value'] );
		
		return $variant;
	}
	
	public function get_doc_link( $info ) {
		return ( $docurl = wpfval( $info, 'docurl' ) ) ? sprintf(
			'<div class="wpf-opt-doc">
                        <a href="%1$s" title="%2$s" target="_blank"><i class="far fa-question-circle"></i></a>
                    </div>',
			esc_url_raw( $docurl ),
			esc_attr__( "Read the documentation", "wpforo" )
		) : '<div class="wpf-opt-doc">&nbsp;</div>';
	}
	
	public function header( $group ) {
		if( ! $info = wpfval( $this->info->core, $group ) ) return;
		printf(
			'<div class="wpf-opt-row"><div class="wpf-opt-intro">%1$s %2$s</div>%3$s</div>',
			wpfval( $info, 'icon' ),
			wpfval( $info, 'description' ),
			$this->get_doc_link( $info )
		);
	}
	
	private function __form_field( $group, $name, $value = null ) {
		if( ! $info = $this->get_option_info( $group, $name ) ) return '';
		if( is_null( $value ) ) $value = wpfval( $this->$group, $name );
		$type = wpfval( $info, 'type' );
		if( ! $type ) $type = 'text';
		switch( $type ) {
			case 'radio'       :
				return $this->radio( $group, $name, $value );
			case 'checkbox'    :
				return $this->checkbox( $group, $name, $value );
			case 'textarea'    :
				return $this->textarea( $group, $name, $value );
			case 'textarea_\n' :
				return $this->textarea( $group, $name, implode( "\n", $value ) );
			case 'textarea_|'  :
				return $this->textarea( $group, $name, implode( "|", $value ) );
			case 'textarea_,'  :
				return $this->textarea( $group, $name, implode( ",", $value ) );
			case 'text_\n' :
				return $this->other( $group, $name, implode( "\n", $value ) );
			case 'text_|'  :
				return $this->other( $group, $name, implode( "|", $value ) );
			case 'text_,'  :
				return $this->other( $group, $name, implode( ",", $value ) );
			case 'select'      :
				return $this->select( $group, $name, $value );
			case 'multiselect'      :
				return $this->select( $group, $name, $value, true );
			case 'image' :
				return $this->image( $group, $name, $value );
			case 'cover' :
				return $this->cover( $group, $name, $value );
			case 'richeditor'  :
				return $this->richeditor( $group, $name, $value );
			case 'ratinglevels':
				return $this->ratinglevels( $group );
			case 'wrap'        :
				return $this->wrap( $group, $name );
			default            :
				return $this->other( $group, $name, $value, $type );
		}
	}
	
	public function form_field( $group, $name, $value = null ) {
		echo $this->__form_field( $group, $name, $value );
	}
	
	private function radio( $group, $name, $value ) {
		if( ! $info = $this->get_option_info( $group, $name ) ) return '';
		$variants    = $this->get_option_variants( $group, $name, $value );
		$option_name = $group . '[' . $name . ']';
		$inputs      = '';
		foreach( $variants as $variant ) {
			$inputs .= sprintf(
				'<input type="radio" id="%1$s" name="%2$s" value="%3$s" %4$s %5$s> <label for="%1$s" title="%6$s">%7$s</label>',
				sanitize_title( $option_name . '_' . $variant['label'] ),
				esc_attr( $option_name ),
				esc_attr( $variant['value'] ),
				checked( $variant['checked'], true, false ),
				disabled( $variant['disabled'], true, false ),
				esc_attr( $variant['title'] ),
				$variant['label']
			);
		}
		
		return sprintf(
			'<div class="wpf-opt-row" data-wpf-opt="%1$s">
                <div class="wpf-opt-name">
                    <label>%2$s</label>
                    <p class="wpf-desc">%3$s</p>
                </div>
                <div class="wpf-opt-input">
                    <div class="wpf-switch-field">%4$s</div>
                </div>
                %5$s
            </div>',
			esc_attr( $name ),
			$info["label"],
			$info["description"],
			$inputs,
			$this->get_doc_link( $info )
		);
	}
	
	private function checkbox( $group, $name, $values ) {
		if( ! $info = $this->get_option_info( $group, $name ) ) return '';
		$variants    = $this->get_option_variants( $group, $name, $values );
		$option_name = $group . '[' . $name . '][]';
		$inputs      = '';
		foreach( $variants as $variant ) {
			$inputs .= sprintf(
				'<div class="wpf-opt-checkbox-wrap"><input type="checkbox" id="%1$s" name="%2$s" value="%3$s" %4$s><label for="%1$s">%5$s</label></div>',
				sanitize_title( $option_name . '_' . $variant['label'] ),
				esc_attr( $option_name ),
				esc_attr( $variant['value'] ),
				checked( $variant['checked'], true, false ),
				$variant['label']
			);
		}
		
		return sprintf(
			'<div class="wpf-opt-row" data-wpf-opt="%1$s">
                <div class="wpf-opt-name">
                    <label>%2$s</label>
                    <p class="wpf-desc">%3$s</p>
                </div>
                <div class="wpf-opt-input">%4$s</div>
               %5$s
            </div>',
			esc_attr( $name ),
			$info["label"],
			$info["description"],
			$inputs,
			$this->get_doc_link( $info )
		);
	}
	
	private function wrap( $group, $name ) {
		if( ! $info = $this->get_option_info( $group, $name ) ) return '';
		$inputs = '';
		foreach( $info['options_keys'] as $key ) {
			$inputs .= $this->__form_field( $group, $key );
		}
		
		return sprintf(
			'<div class="wpf-opt-row" data-wpf-opt="%1$s">
                <div class="wpf-opt-name">
                    <label>%2$s</label>
                    <p class="wpf-desc">%3$s</p>
                </div>
                <div class="wpf-opt-input">%4$s</div>
                %5$s
            </div>',
			esc_attr( $name ),
			$info["label"],
			$info["description"],
			$inputs,
			$this->get_doc_link( $info )
		);
	}
	
	private function other( $group, $name, $value, $type = 'text' ) {
		if( ! $info = $this->get_option_info( $group, $name ) ) return '';
		$option_name = $group . '[' . $name . ']';
		$after       = $step = '';
		$min         = wpfval( $info, 'min' );
		$max         = wpfval( $info, 'max' );
		if( $type === 'float' ) {
			$type = 'number';
			$step = 'step="0.1"';
		} elseif( in_array( $type, [ 'terabyte', 'gigabyte', 'megabyte', 'kilobyte', 'byte' ], true ) ) {
			$_min = 1;
			$_max = null;
			switch( $type ) {
				case 'terabyte':
					$value = intval( $value ) / 1024 / 1024 / 1024 / 1024;
					$after = __( 'TB', 'wpforo' );
					$_max  = $this->_SERVER['maxs_min'] / 1024 / 1024 / 1024 / 1024;
				break;
				case 'gigabyte':
					$value = intval( $value ) / 1024 / 1024 / 1024;
					$after = __( 'GB', 'wpforo' );
					$_max  = $this->_SERVER['maxs_min'] / 1024 / 1024 / 1024;
				break;
				case 'megabyte':
					$value = intval( $value ) / 1024 / 1024;
					$after = __( 'MB', 'wpforo' );
					$_max  = $this->_SERVER['maxs_min'] / 1024 / 1024;
				break;
				case 'kilobyte':
					$value = intval( $value ) / 1024;
					$after = __( 'KB', 'wpforo' );
					$_max  = $this->_SERVER['maxs_min'] / 1024;
				break;
				case 'byte':
					$value = intval( $value );
					$after = __( 'B', 'wpforo' );
					$_max  = $this->_SERVER['maxs_min'];
				break;
			}
			
			if( $min ) {
				if( $min > $this->_SERVER['maxs_min'] ) $min = $this->_SERVER['maxs_min'];
			} else {
				$min = $_min;
			}
			
			if( $max ) {
				if( $max > $this->_SERVER['maxs_min'] ) $max = $this->_SERVER['maxs_min'];
			} else {
				$max = $_max;
			}
			
			$type = 'number';
			if( ! wpfkey( $info, 'info' ) ) {
				$info['info'] = __( 'Min Allowed Size is ' ) . $min . $after . '<br/>' . __( 'Max Allowed Size is ' ) . $max . $after . '<br/>' . __(
						'Server "upload_max_filesize" is '
					) . $this->_SERVER['upload_max_filesize_human'] . '<br/>' . __( 'Server "post_max_size" is ' ) . $this->_SERVER['post_max_size_human'];
			}
		} elseif( $type === 'minute' ) {
			$type  = 'number';
			$value = intval( $value ) / 60;
			if( ! wpfkey( $info, 'min' ) ) $min = 1;
			$after = __( 'Minute', 'wpforo' );
		}
		$attr_min = $attr_max = '';
		if( in_array( $type, [ 'number', 'range', 'minute' ], true ) ) {
			if( ! is_null( $min ) ) $attr_min = sprintf( 'min="%1$d"', $min );
			if( ! is_null( $max ) ) $attr_max = sprintf( 'max="%1$d"', $max );
		} elseif( in_array( $type, [ 'date', 'datetime-local', 'month', 'time', 'week' ], true ) ) {
			if( ! is_null( $min ) ) $attr_min = sprintf( 'min="%1$s"', $min );
			if( ! is_null( $max ) ) $attr_max = sprintf( 'max="%1$s"', $max );
		} else {
			if( ! is_null( $min ) ) $attr_min = sprintf( 'minlength="%1$d"', $min );
			if( ! is_null( $max ) ) $attr_max = sprintf( 'maxlength="%1$d"', $max );
		}
		
		return sprintf(
			'<div class="wpf-opt-row" data-wpf-opt="%1$s">
                %2$s
                <div class="wpf-opt-input">
                    <input id="%3$s" type="%4$s" value="%5$s" name="%6$s" placeholder="%7$s" %8$s %9$s %10$s>
                    <span class="wpf-opt-input-after">%11$s</span>
                    <p class="wpf-info">%12$s</p>
                </div>
                %13$s
            </div>',
			esc_attr( $name ),
			( $info["label"] || $info["description"] ? sprintf(
				'<div class="wpf-opt-name">
                        <label for="%1$s">%2$s</label>
                        <p class="wpf-desc">%3$s</p>
                    </div>',
				sanitize_title( $option_name ),
				$info["label"],
				$info["description"]
			) : '' ),
			sanitize_title( $option_name ),
			$type,
			esc_attr( $value ),
			esc_attr( $option_name ),
			esc_attr( (string) wpfval( $info, 'placeholder' ) ),
			$step,
			$attr_min,
			$attr_max,
			$after,
			wpfval( $info, 'info' ),
			$this->get_doc_link( $info )
		);
	}
	
	private function textarea( $group, $name, $value ) {
		if( ! $info = $this->get_option_info( $group, $name ) ) return '';
		$option_name = $group . '[' . $name . ']';
		$min         = wpfval( $info, 'min' );
		$max         = wpfval( $info, 'max' );
		$attr_min    = $attr_max = '';
		if( ! is_null( $min ) ) $attr_min = sprintf( 'minlength="%1$d"', $min );
		if( ! is_null( $max ) ) $attr_max = sprintf( 'maxlength="%1$d"', $max );
		
		return sprintf(
			'<div class="wpf-opt-row wpf-opt-row-textarea" data-wpf-opt="%1$s">
                    <div class="wpf-opt-name">
                        <label for="%2$s">%3$s</label>
                        <p class="wpf-desc">%4$s</p>
                    </div>
                    <div class="wpf-opt-input">
                        <textarea id="%2$s" name="%5$s" %6$s %7$s placeholder="%8$s">%9$s</textarea>
                    </div>
                    %10$s
                </div>',
			esc_attr( $name ),
			sanitize_title( $option_name ),
			$info["label"],
			$info["description"],
			$option_name,
			$attr_min,
			$attr_max,
			esc_attr( (string) wpfval( $info, 'placeholder' ) ),
			$value,
			$this->get_doc_link( $info )
		);
	}
	
	private function richeditor( $group, $name, $value ) {
		if( ! $info = $this->get_option_info( $group, $name ) ) return '';
		$option_name = $group . '[' . $name . ']';
		ob_start();
		wp_editor( wpautop( wp_unslash( $value ) ), sanitize_title( $option_name ), [
			'teeny'          => false,
			'media_buttons'  => (bool) wpfval( $info, 'media_buttons' ),
			'textarea_rows'  => 8,
			'tinymce'        => [ 'wpautop' => false, 'setup' => 'wpforo_admin_tinymce_setup' ],
			'quicktags'      => [ 'buttons' => 'strong,em,link,block,del,ins,img,ul,ol,li,code,close' ],
			'default_editor' => 'tinymce',
			'textarea_name'  => $option_name,
		] );
		$wp_editor = ob_get_clean();
		
		return sprintf(
			'<div class="wpf-opt-row %1$s" data-wpf-opt="%2$s">
                <div class="wpf-opt-name">
                    <label for="%3$s">%4$s</label>
                    <p class="wpf-desc">%5$s</p>
                </div>
                <div class="wpf-opt-input">%6$s</div>
                %7$s
            </div>',
			( wpfval( $info, 'fullwidth' ) ? 'wpf-opt-row-fullwidth' : '' ),
			esc_attr( $name ),
			sanitize_title( $option_name ),
			$info["label"],
			$info["description"],
			$wp_editor,
			$this->get_doc_link( $info )
		);
	}
	
	private function select( $group, $name, $value, $multi = false ) {
		if( ! $info = $this->get_option_info( $group, $name ) ) return '';
		$variants    = $this->get_option_variants( $group, $name, $value );
		$option_name = $group . '[' . $name . ']' . ( $multi ? '[]' : '' );
		$options     = '';
		foreach( $variants as $variant ) {
			$options .= sprintf(
				'<option value="%1$s" title="%2$s" %3$s %4$s>%5$s</option>',
				esc_attr( $variant['value'] ),
				esc_attr( $variant['title'] ),
				selected( $variant['selected'], true, false ),
				disabled( $variant['disabled'], true, false ),
				esc_html( $variant['label'] )
			);
		}
		
		return sprintf(
			'<div class="wpf-opt-row wpf-opt-row-select" data-wpf-opt="%1$s">
                <div class="wpf-opt-name">
                    <label for="%2$s">%3$s</label>
                    <p class="wpf-desc">%4$s</p>
                </div>
                <div class="wpf-opt-input">
                    <div class="wpf-switch-field">
                        <select id="%2$s" name="%5$s" %6$s>%7$s</select>
                    </div>
                </div>
                %8$s
            </div>',
			esc_attr( $name ),
			sanitize_title( $option_name ),
			$info["label"],
			$info["description"],
			esc_attr( $option_name ),
			( $multi ? 'multiple' : '' ),
			$options,
			$this->get_doc_link( $info )
		);
	}
	
	private function image( $group, $name, $value ) {
		if( ! $info = $this->get_option_info( $group, $name ) ) return '';
		$option_name = $group . '[' . $name . ']';
		
		return sprintf(
			'<div class="wpf-opt-row wpf-opt-row-image" data-wpf-opt="%1$s">
                <div class="wpf-opt-name">
                    <label for="%2$s">%3$s</label>
                    <p class="wpf-desc">%4$s</p>
                </div>
                <div class="wpf-opt-input">
                    <input type="file" accept="image/png,image/x-png,image/gif,image/jpeg,image/webp" id="%2$s" name="%5$s">
                    %6$s
                </div>
                %7$s
            </div>',
			esc_attr( $name ),
			sanitize_title( $option_name ),
			$info["label"],
			$info["description"],
			esc_attr( $option_name ),
			$value ? sprintf( '<img src="%1$s" alt="">', $value ) : '',
			$this->get_doc_link( $info )
		);
	}
	
	private function cover( $group, $name, $value ) {
		if( ! $info = $this->get_option_info( $group, $name ) ) return '';
		$option_name = $group . '[' . $name . ']';
		
		return sprintf(
			'<div class="wpf-opt-row wpf-opt-row-cover" data-wpf-opt="%1$s" style="flex-direction: column;">
                <div class="wpf-opt-name">
                    <label for="%2$s">%3$s</label>
                    <p class="wpf-desc">%4$s</p>
                </div>
                <div class="wpf-opt-input">
                    <div data-name="%5$s" class="wpf-cover" style="background-image: url(\'%6$s\');">
                        <div class="wpf-edit-cover" wpf-tooltip="Change Cover Image" wpf-tooltip-position="top" wpf-tooltip-size="small"><i class="fas fa-images"></i></div>
                    </div>
                </div>
                %7$s
            </div>',
			esc_attr( $name ),
			sanitize_title( $option_name ),
			$info["label"],
			$info["description"],
			esc_attr( $option_name ),
			( $value ?: $this->_profiles['default_cover'] ),
			$this->get_doc_link( $info )
		);
	}
	
	private function ratinglevels( $group ) {
		ob_start();
		?>
        <div class="wpf-opt-row" data-wpf-opt="levels">
            <table id="wpf-rating-table" class="wpforo_settings_table" style="margin-top: 5px;">
                <tbody>
                <tr>
                    <th style="width:10%;"><?php _e( 'Rating Level', 'wpforo' ); ?></th>
                    <th style="width:15%;"><?php _e( 'Min Number of Points', 'wpforo' ); ?></th>
                    <th style="width:20%;"><?php _e( 'Member Title', 'wpforo' ); ?></th>
                    <th style="text-align:center;width:10%;"><?php _e( 'Short Badge', 'wpforo' ); ?></th>
                    <th style="text-align:center;width:15%;"><?php _e( 'Full Badge', 'wpforo' ); ?></th>
                    <th style="text-align:center;width:10%;"><?php _e( 'Rating Color', 'wpforo' ); ?></th>
                    <th style="width:20%;"><?php _e( 'Rating Icon', 'wpforo' ); ?> | <a
                                href="https://fontawesome.com/icons?d=gallery&p=2&m=free" target="_blank"
                                style="text-decoration:none;"><?php _e( 'More', 'wpforo' ); ?>&raquo;</a></th>
                </tr>
				<?php $levels = WPF()->member->levels(); ?>
				<?php foreach( $levels as $level ): ?>
                    <tr>
                        <td><h4><?php _e( 'Level', 'wpforo' ); ?><?php echo esc_html( $level ) ?></h4></td>
                        <td><label>
                                <input type="number" value="<?php echo WPF()->member->rating( $level, 'points' ) ?>"
                                       name="<?php echo $group ?>[levels][<?php echo esc_attr( $level ) ?>][points]"
                                       placeholder="<?php _e( 'Number of Posts', 'wpforo' ); ?>">
                            </label></td>
                        <td style="text-align:center;"><label>
                                <input type="text"
                                       value="<?php echo WPF()->member->rating( $level, 'title' ) ?>"
                                       name="<?php echo $group ?>[levels][<?php echo esc_attr( $level ) ?>][title]"
                                       placeholder="<?php _e( 'Custom Title', 'wpforo' ); ?>">
                            </label></td>
                        <td style="text-align:center;">
                            <div class="wpf-badge-short wpf-badge-level-<?php echo esc_attr( $level ) ?>"
                                 style="background-color:<?php echo WPF()->member->rating( $level, 'color' ) ?>;"><?php echo WPF()->member->rating_badge( $level ); ?></div>
                        </td>
                        <td style="text-align:center;">
                            <div class="wpf-badge-full wpf-badge-level-<?php echo esc_attr( $level ) ?>"
                                 style="color:<?php echo WPF()->member->rating( $level, 'color' ) ?>;"><?php echo WPF()->member->rating_badge( $level, 'full' ); ?></div>
                        </td>
                        <td style="text-align:center;"><label>
                                <input type="color"
                                       value="<?php echo WPF()->member->rating( $level, 'color' ) ?>"
                                       name="<?php echo $group ?>[levels][<?php echo esc_attr( $level ) ?>][color]"
                                       placeholder="<?php _e( 'Color', 'wpforo' ); ?>">
                            </label></td>
                        <td>
                            <div>
                                <label>
                                    <input type="text" value="<?php echo WPF()->member->rating( $level, 'icon' ) ?>"
                                           name="<?php echo $group ?>[levels][<?php echo esc_attr( $level ) ?>][icon]"
                                           placeholder="<?php _e( 'Badge Icon', 'wpforo' ); ?>">
                                </label>
                            </div>
                        </td>
                    </tr>
				<?php endforeach; ?>
                </tbody>
            </table>
        </div>
		<?php
		return ob_get_clean();
	}
	
	public function get_variants_usergroups( $excludeids = [], $secondary = false ) {
		if( ! WPF()->is_installed() ) return [];
		$variants = [];
		if( $secondary ) {
			$groups = WPF()->usergroup->get_secondary_groups();
		} else {
			$groups = WPF()->usergroup->get_usergroups();
		}
		foreach( $groups as $group ) {
			if( in_array( (int) $group['groupid'], (array) $excludeids, true ) ) continue;
			$variants[] = [ 'value' => (int) $group['groupid'], 'label' => $group['name'] ];
		}
		
		return $variants;
	}
	
	public function init_defaults() {
		$blogname          = get_option( 'blogname', '' );
		$adminemail        = get_option( 'admin_email', '' );
		$max_upload_size   = $this->_SERVER['upload_max_filesize'];
		$max_rating_levels = (int) apply_filters( 'wpforo_max_rating_levels', 10 );
		if( ! $max_upload_size || $max_upload_size > 10485760 ) $max_upload_size = 10485760;
		
		$this->_slugs = [
			'paged'   => 'paged',
			'postid'  => 'postid',
			'topicid' => 'topicid',
		];
		$this->_slugs = (array) apply_filters( 'wpforo_settings_init__slugs', $this->_slugs );
		
		$this->_general  = [
			'admin_bar'      => [ 1, 2 ],
			'wp_date_format' => false,
			'fontawesome'    => 'sitewide',
			'debug_mode'     => false,
			'current_theme'  => '2022',
		];
		$this->_members  = [
			'list_order'       => 'posts__desc',
			'hide_inactive'    => true,
			'members_per_page' => 15,
			'search_type'      => 'search',
		];
		$this->_profiles = [
			'profile'                  => 'default',
			'profile_header'           => false,
			'profile_footer'           => false,
			'url_structure'            => 'nicename',
			'online_status_timeout'    => 240,
			'custom_title_is_on'       => true,
			'default_title'            => 'Member',
			'title_groupids'           => [ 1, 2, 4, 5 ],
			'title_secondary_groupids' => [],
			'mention_nicknames'        => true,
			'avatars'                  => true,
			'custom_avatars'           => true,
			'replace_avatar'           => true,
			'default_cover'            => WPFORO_URL . '/assets/images/profile-bg.jpg',
			'signature'                => true,
		];
		$this->_rating   = [
			'rating'          => true,
			'rating_title'    => true,
			'topic_points'    => 2,
			'post_points'     => 1,
			'like_points'     => 0.5,
			'dislike_points'  => - 0.5,
			'rating_title_ug' => [ 3, 4, 5 ],
			'rating_badge_ug' => [ 1, 2, 3, 4, 5 ],
			'levels'          => [
				0  => [
					'points' => 0,
					'color'  => '#d2d2d2',
					'title'  => 'New Member',
					'icon'   => 'far fa-star-half',
				],
				1  => [
					'points' => 5,
					'color'  => '#4dca5c',
					'title'  => 'Active Member',
					'icon'   => 'fas fa-star',
				],
				2  => [
					'points' => 20,
					'color'  => '#4dca5c',
					'title'  => 'Eminent Member',
					'icon'   => 'fas fa-star',
				],
				3  => [
					'points' => 50,
					'color'  => '#4dca5c',
					'title'  => 'Trusted Member',
					'icon'   => 'fas fa-star',
				],
				4  => [
					'points' => 100,
					'color'  => '#4dca5c',
					'title'  => 'Estimable Member',
					'icon'   => 'fas fa-star',
				],
				5  => [
					'points' => 250,
					'color'  => '#4dca5c',
					'title'  => 'Reputable Member',
					'icon'   => 'fas fa-star',
				],
				6  => [
					'points' => 500,
					'color'  => '#E5D600',
					'title'  => 'Honorable Member',
					'icon'   => 'fas fa-certificate',
				],
				7  => [
					'points' => 750,
					'color'  => '#E5D600',
					'title'  => 'Prominent Member',
					'icon'   => 'fas fa-certificate',
				],
				8  => [
					'points' => 1000,
					'color'  => '#E5D600',
					'title'  => 'Noble Member',
					'icon'   => 'fas fa-certificate',
				],
				9  => [
					'points' => 2500,
					'color'  => '#FF812D',
					'title'  => 'Famed Member',
					'icon'   => 'fas fa-shield-alt',
				],
				10 => [
					'points' => 5000,
					'color'  => '#FF812D',
					'title'  => 'Illustrious Member',
					'icon'   => 'fas fa-shield-alt',
				],
				11 => [
					'points' => 10000,
					'color'  => '#FF812D',
					'title'  => 'Majestic Member',
					'icon'   => 'fas fa-shield-alt',
				],
				12 => [
					'points' => 25000,
					'color'  => '#E04A47',
					'title'  => 'Mythical Member',
					'icon'   => 'fas fa-trophy',
				],
				13 => [
					'points' => 50000,
					'color'  => '#E04A47',
					'title'  => 'Fabled Member',
					'icon'   => 'fas fa-trophy',
				],
				14 => [
					'points' => 100000,
					'color'  => '#E04A47',
					'title'  => 'Legendary Member',
					'icon'   => 'fas fa-trophy',
				],
			],
		];
		if( $max_rating_levels && $max_rating_levels < 14 ) {
			$this->_rating['levels'] = array_slice( $this->_rating['levels'], 0, ( $max_rating_levels + 1 ), true );
			if( $max_rating_levels <= 10 ) {
				$this->_rating['levels'][10]['icon']  = 'fas fa-trophy';
				$this->_rating['levels'][10]['color'] = '#E04A47';
			}
		}
		$this->_authorization = [
			'user_register'                     => true,
			'user_register_email_confirm'       => true,
			'manually_approval'                 => false,
			'manually_approval_contact_form'    => false,
			'role_synch'                        => true,
			'user_delete_method'                => 'soft',
			'send_email_after_user_delete'      => true,
			'use_our_register_url'              => true,
			'use_our_login_url'                 => true,
			'use_our_lostpassword_url'          => true,
			'login_url'                         => '',
			'register_url'                      => '',
			'lost_password_url'                 => '',
			'redirect_url_after_login'          => '',
			'redirect_url_after_register'       => '',
			'redirect_url_after_confirm_sbscrb' => '',
			'fb_api_id'                         => '',
			'fb_api_secret'                     => '',
			'fb_login'                          => false,
			'fb_lb_on_lp'                       => true,
			'fb_lb_on_rp'                       => true,
			'fb_redirect'                       => 'profile',
			'fb_redirect_url'                   => '',
		];
		$this->_recaptcha     = [
			'site_key'          => '',
			'secret_key'        => '',
			'theme'             => 'light',
			'topic_editor'      => true,
			'post_editor'       => true,
			'wpf_login_form'    => true,
			'wpf_reg_form'      => true,
			'wpf_lostpass_form' => true,
			'login_form'        => false,
			'reg_form'          => false,
			'lostpass_form'     => false,
		];
		$this->_buddypress    = [
			'activity'     => true,
			'notification' => true,
			'forum_tab'    => true,
		];
		$this->_um            = [
			'notification' => true,
			'forum_tab'    => true,
		];
		$this->_board         = [
			'under_construction' => false,
			'cache'              => true,
			'url_structure'      => 'full',
		];
		$this->_forums        = [
			'layout_extended_intro_topics_toggle' => true,
			'layout_extended_intro_topics_count'  => 3,
			'layout_extended_intro_topics_length' => 45,
			'layout_simplified_add_topic_button'  => true,
			'layout_qa_intro_topics_toggle'       => true,
			'layout_qa_intro_topics_count'        => 3,
			'layout_qa_intro_topics_length'       => 90,
			'layout_qa_add_topic_button'          => true,
			'layout_threaded_intro_topics_toggle' => false,
			'layout_threaded_display_subforums'   => true,
			'layout_threaded_filter_buttons'      => true,
			'layout_threaded_add_topic_button'    => true,
			'layout_threaded_intro_topics_count'  => 10,
			'layout_threaded_intro_topics_length' => 0,
		];
		$this->_topics        = [
			'layout_extended_intro_posts_toggle' => true,
			'layout_extended_intro_posts_count'  => 4,
			'layout_extended_intro_posts_length' => 50,
			'layout_qa_posts_per_page'           => 15,
			'layout_qa_comments_limit_count'     => 3,
			'layout_qa_first_post_reply'         => true,
			'layout_threaded_posts_per_page'     => 5,
			'layout_threaded_nesting_level'      => 5,
			'layout_threaded_first_post_reply'   => false,
			'include_subforums_topics'           => false,
			'topics_per_page'                    => 10,
			'posts_per_page'                     => 15,
			'search_max_results'                 => 100,
			'union_first_post'                   => [
				1 => false,
				2 => false,
				3 => true,
				4 => false,
			],
			'recent_posts_type'                  => 'topics',
			'topic_head'                         => true,
			'topic_head_expanded'                => false,
		];
		$this->_posting       = [
			'qa_display_answer_editor'      => true,
			'qa_comments_rich_editor'       => false,
			'threaded_reply_rich_editor'    => true,
			'topic_title_min_length'        => 1,
			'topic_title_max_length'        => 0,
			'topic_body_min_length'         => 2,
			'topic_body_max_length'         => 0,
			'post_body_min_length'          => 2,
			'post_body_max_length'          => 0,
			'comment_body_min_length'       => 2,
			'comment_body_max_length'       => 0,
			'edit_own_topic_durr'           => 300,
			'delete_own_topic_durr'         => 300,
			'edit_own_post_durr'            => 300,
			'delete_own_post_durr'          => 300,
			'edit_topic'                    => true,
			'edit_post'                     => true,
			'edit_log_display_limit'        => 0,
			'is_preview_on'                 => true,
			'is_draft_on'                   => true,
			'auto_draft_interval'           => 30000,
			'max_drafts_per_page'           => 3,
			'max_upload_size'               => $max_upload_size,
			'attachs_to_medialib'           => true,
			'topic_editor_toolbar_location' => 'top',
			'reply_editor_toolbar_location' => 'top',
			'content_do_shortcode'          => false,
			'extra_html_tags'               => 'embed(src width height name pluginspage type wmode allowFullScreen allowScriptAccess flashVars),',
		];
		$this->_components    = [
			'admin_cp'         => true,
			'page_title'       => true,
			'top_bar'          => true,
			'top_bar_search'   => true,
			'sidebar_location' => 'default',
			'breadcrumb'       => true,
			'footer'           => true,
			'footer_stat'      => true,
			'copyright'        => true,
		];
		$this->_styles        = [
			'font_size_forum'        => 17,
			'font_size_topic'        => 16,
			'font_size_post_content' => 15,
			'custom_css'             => "#wpforo-wrap{\r\nfont-size: 13px; width: 100%; padding: 10px 0; margin: 0px;\r\n}\r\n",
			'color_style'            => 'default',
			'color_styles'           => [
				'default' => [
					0  => '#000000',
					1  => '#ffffff',
					2  => '#333333',
					3  => '#555555',
					4  => '#666666',
					5  => '#777777',
					6  => '#999999',
					7  => '#cccccc',
					8  => '#e6e6e6',
					9  => '#f5f5f5',
					10 => '#dadada',
					11 => '#659fbe',
					12 => '#43a6df',
					13 => '#72ccfc',
					14 => '#0099cc',
					15 => '#3f7796',
					16 => '#4a8eb3',
					17 => '#dff4ff',
					18 => '#fafafa',
					20 => '#ff812d',
					30 => '#4dca5c',
					31 => '#00a636',
					32 => '#86ba4c',
					33 => '#6fa634',
					40 => '#ff9595',
					41 => '#ff7575',
					42 => '#f46464',
				],
				'red'     => [
					0  => '#000000',
					1  => '#ffffff',
					2  => '#333333',
					3  => '#555555',
					4  => '#666666',
					5  => '#777777',
					6  => '#999999',
					7  => '#cccccc',
					8  => '#e6e6e6',
					9  => '#f5f5f5',
					10 => '#dadada',
					11 => '#E0141E',
					12 => '#EE1A26',
					13 => '#FC979C',
					14 => '#E0141E',
					15 => '#99262B',
					16 => '#D61319',
					17 => '#FFF7F7',
					18 => '#fafafa',
					20 => '#30B2A7',
					30 => '#4dca5c',
					31 => '#00a636',
					32 => '#86ba4c',
					33 => '#6fa634',
					40 => '#ff9595',
					41 => '#ff7575',
					42 => '#f46464',
				],
				'green'   => [
					0  => '#000000',
					1  => '#ffffff',
					2  => '#333333',
					3  => '#555555',
					4  => '#666666',
					5  => '#777777',
					6  => '#999999',
					7  => '#cccccc',
					8  => '#e6e6e6',
					9  => '#f5f5f5',
					10 => '#dadada',
					11 => '#6EA500',
					12 => '#649E2D',
					13 => '#8DCE0C',
					14 => '#447714',
					15 => '#5A7F10',
					16 => '#6EA500',
					17 => '#F8FCEF',
					18 => '#fafafa',
					20 => '#ff812d',
					30 => '#4dca5c',
					31 => '#00a636',
					32 => '#FF812D',
					33 => '#F47222',
					40 => '#ff9595',
					41 => '#ff7575',
					42 => '#f46464',
				],
				'orange'  => [
					0  => '#000000',
					1  => '#ffffff',
					2  => '#333333',
					3  => '#555555',
					4  => '#666666',
					5  => '#777777',
					6  => '#999999',
					7  => '#cccccc',
					8  => '#e6e6e6',
					9  => '#f5f5f5',
					10 => '#dadada',
					11 => '#E0762F',
					12 => '#FF6600',
					13 => '#FC9958',
					14 => '#F26000',
					15 => '#AA4F12',
					16 => '#F26000',
					17 => '#FFF4ED',
					18 => '#fafafa',
					20 => '#ff812d',
					30 => '#4dca5c',
					31 => '#00a636',
					32 => '#86ba4c',
					33 => '#6fa634',
					40 => '#ff9595',
					41 => '#ff7575',
					42 => '#f46464',
				],
				'grey'    => [
					0  => '#000000',
					1  => '#ffffff',
					2  => '#333333',
					3  => '#343434',
					4  => '#666666',
					5  => '#777777',
					6  => '#999999',
					7  => '#cccccc',
					8  => '#e6e6e6',
					9  => '#f5f5f5',
					10 => '#dadada',
					11 => '#888888',
					12 => '#666666',
					13 => '#7EEA8D',
					14 => '#777777',
					15 => '#333333',
					16 => '#555555',
					17 => '#DFF4FF',
					18 => '#fafafa',
					20 => '#FF812D',
					30 => '#4dca5c',
					31 => '#00a636',
					32 => '#86ba4c',
					33 => '#6fa634',
					40 => '#ff9595',
					41 => '#ff7575',
					42 => '#f46464',
				],
				'dark'    => [
					0  => '#000000',
					1  => '#141414',
					2  => '#bbbbbb',
					3  => '#000000',
					4  => '#666666',
					5  => '#bcbcbc',
					6  => '#999999',
					7  => '#585858',
					8  => '#727272',
					9  => '#323232',
					10 => '#dadada',
					11 => '#888888',
					12 => '#33779b',
					13 => '#7EEA8D',
					14 => '#777777',
					15 => '#E0E0E0',
					16 => '#CECECE',
					17 => '#33779b',
					18 => '#666666',
					20 => '#FF812D',
					30 => '#4dca5c',
					31 => '#00a636',
					32 => '#86ba4c',
					33 => '#6fa634',
					40 => '#ff9595',
					41 => '#ff7575',
					42 => '#f46464',
				],
			],
		];
		$this->_tags          = [
			'max_per_topic' => 5,
			'per_page'      => 100,
			'length'        => 25,
			'suggest_limit' => 5,
			'lowercase'     => false,
		];
		$this->_email         = [
			'from_name'                                          => $blogname . ' - ' . __( 'Forum', 'wpforo' ),
			'from_email'                                         => $adminemail,
			'admin_emails'                                       => (array) $adminemail,
			'new_topic_notify'                                   => true,
			'new_reply_notify'                                   => false,
			'disable_new_user_admin_notification'                => true,
			'report_email_subject'                               => __( "Forum Post Report", 'wpforo' ),
			'report_email_message'                               => __( "<strong>Report details:</strong>\n Reporter: [reporter], <br>\n Message: [message],<br>\n <br>\n [post_url]", 'wpforo' ),
			'overwrite_new_user_notification_admin'              => true,
			'wp_new_user_notification_email_admin_subject'       => __( "[blogname] New User Registration", 'wpforo' ),
			'wp_new_user_notification_email_admin_message'       => __( "New user registration on your site [blogname]:\n\nUsername: [user_login]\n\nEmail: [user_email]\n", 'wpforo' ),
			'after_user_delete_notification_email_admin_subject' => __( "User Self-Deleted Their Account", 'wpforo' ),
			'after_user_delete_notification_email_admin_message' => __(
				"This is to notify you that a user has successfully deleted their account from the website. Below are the details: \n\nUser ID: [user_userid]\n\nUsername: [user_login]\n\nEmail: [user_email]\n\nDeletion Date and Time: [datetime] \n",
				'wpforo'
			),
			'overwrite_new_user_notification'                    => true,
			'wp_new_user_notification_email_subject'             => __( "[blogname] Your username and password info", 'wpforo' ),
			'wp_new_user_notification_email_message'             => __( "Username: [user_login]\n\nTo set your password, visit the following address:\n\n[set_password_url]\n\n", 'wpforo' ),
			'overwrite_reset_password_email'                     => true,
			'reset_password_email_message'                       => __(
				"Hello! \n\n You asked us to reset your password for your account using the email address [user_login]. \n\n If this was a mistake, or you didn't ask for a password reset, just ignore this email and nothing will happen. \n\n To reset your password, visit the following address: \n\n [reset_password_url] \n\n Thanks!",
				'wpforo'
			),
			'after_user_approve_email_subject'                   => __( '[blogname] - Your Account Has been Approved', 'wpforo' ),
			'after_user_approve_email_message'                   => __(
				'Hi [user_login] , Thank you for your registration. Your account has been approved. You can login here: [login_link]',
				'wpforo'
			),
		];
		$this->_subscriptions = [
			'subscribe_confirmation'               => true,
			'subscribe_checkbox_on_post_editor'    => true,
			'subscribe_checkbox_default_status'    => false,
			'user_mention_notify'                  => true,
			'user_following_notify'                => true,
			'confirmation_email_subject'           => __( "Please confirm subscription to [entry_title]", 'wpforo' ),
			'confirmation_email_message'           => __(
				"Hello [user_display_name]!<br>\n Thank you for subscribing.<br>\n This is an automated response.<br>\n We are glad to inform you that after confirmation you will get updates from - [entry_title].<br>\n Please click on link below to complete this step.<br>\n [confirm_link]",
				'wpforo'
			),
			'new_topic_notification_email_subject' => __( "New Topic", 'wpforo' ),
			'new_topic_notification_email_message' => __(
				"Hello [user_display_name]!<br>\n New topic has been created on your subscribed forum - [forum_link].\n <br><br>\n <strong>[topic_title]</strong>\n <blockquote>\n [topic_body]\n </blockquote>\n <br><hr>\n If you want to unsubscribe from this forum please use the link below.<br>\n [unsubscribe_link]",
				'wpforo'
			),
			'new_post_notification_email_subject'  => __( "New Reply", 'wpforo' ),
			'new_post_notification_email_message'  => __(
				"Hello [user_display_name]!<br>\n New reply has been posted on your subscribed topic - [topic_link].\n <br><br>\n <strong>[post_title]</strong>\n <blockquote >\n [post_body]\n </blockquote>\n <br><hr>\n If you want to unsubscribe from this topic please use the link below.<br>\n [unsubscribe_link]",
				'wpforo'
			),
			'user_mention_email_subject'           => __( "You have been mentioned in forum post", 'wpforo' ),
			'user_mention_email_message'           => __(
				"Hi [user_display_name]! <br>\n\n You have been mentioned in a post on \"[topic_title]\" by [owner_display_name].<br/><br/>\n\n [post_link]",
				'wpforo'
			),
			'user_following_email_subject'         => __( "New Post from [owner_display_name]", 'wpforo' ),
			'user_following_email_message'         => __(
				"Hello [user_display_name]!<br>\n New content has been posted by [owner_display_name] you are following.\n <br><br>\n <strong>[post_link]</strong>\n <blockquote >\n [post_body]\n </blockquote>\n <br><hr>\n If you want to unfollow this user please use the link below.<br>\n [unsubscribe_link]",
				'wpforo'
			),
		];
		$this->_notifications = [
			'notifications'      => true,
			'notifications_live' => false,
			'notifications_bar'  => true,
		];
		$this->_logging       = [
			'view_logging'                  => true,
			'track_logging'                 => true,
			'goto_unread'                   => true,
			'goto_unread_button'            => false,
			'display_forum_current_viewers' => true,
			'display_topic_current_viewers' => true,
			'display_recent_viewers'        => true,
			'display_admin_viewers'         => true,
		];
		$this->_seo           = [
			'seo_title'       => true,
			'seo_meta'        => true,
			'seo_profile'     => false,
			'forums_sitemap'  => true,
			'topics_sitemap'  => true,
			'members_sitemap' => true,
			'dofollow'        => [],
			'noindex'         => [],
		];
		$this->_antispam      = [
			'spam_filter'                      => true,
			'spam_user_ban'                    => false,
			'should_unapprove_after_report'    => false,
			'spam_filter_level_topic'          => mt_rand( 30, 60 ),
			'spam_filter_level_post'           => mt_rand( 30, 60 ),
			'new_user_max_posts'               => 3,
			'unapprove_post_if_user_is_new'    => false,
			'min_number_posts_to_attach'       => 0,
			'min_number_posts_to_edit_account' => 3,
			'min_number_posts_to_link'         => 0,
			'limited_file_ext'                 => [ 'pdf', 'doc', 'docx', 'txt', 'htm', 'html', 'rtf', 'xml', 'xls', 'xlsx', 'zip', 'rar', 'tar', 'gz', 'bzip', '7z' ],
			'spam_file_scanner'                => true,
			'exclude_file_ext'                 => [ 'pdf', 'doc', 'docx', 'txt' ],
		];
		$this->_akismet       = [
			'akismet' => true,
		];
		$this->_rss           = [
			'feed'         => true,
			'feed_general' => true,
			'feed_forum'   => true,
			'feed_topic'   => true,
		];
		$this->_social        = [
			'sb'                 => [ 'fb' => true, 'tw' => true, 'wapp' => true, 'lin' => false, 'vk' => false, 'ok' => false, 'gg' => false ],
			'sb_on'              => true,
			'sb_toggle_on'       => false,
			'sb_style'           => 'grey',
			'sb_type'            => 'icon',
			'sb_toggle'          => 4,
			'sb_toggle_type'     => 'collapsed',
			'sb_icon'            => 'mixed',
			'sb_location'        => [ 'top' => false, 'bottom' => true ],
			'sb_location_toggle' => 'top',
		];
		$this->_legal         = [
			'contact_page_url'        => '',
			'checkbox_terms_privacy'  => false,
			'checkbox_email_password' => true,
			'page_terms'              => '',
			'page_privacy'            => '',
			'checkbox_forum_privacy'  => false,
			'forum_privacy_text'      => wpforo_get_file_content( WPFORO_DIR . '/admin/assets/html/privacy-policy-gdpr.html' ),
			'checkbox_fb_login'       => true,
			'cookies'                 => true,
			'rules_text'              => wpforo_get_file_content( WPFORO_DIR . '/admin/assets/html/simple-forum-rules.html' ),
			'rules_checkbox'          => false,
		];
	}
	
	private function init_base() {
		$this->general                 = wpforo_get_option( 'wpforo_general', $this->_general );
		$this->members                 = wpforo_get_option( 'wpforo_members', $this->_members );
		$this->profiles                = wpforo_get_option( 'wpforo_profiles', $this->_profiles );
		$this->rating                  = wpforo_get_option( 'wpforo_rating', $this->_rating );
		$this->authorization           = wpforo_get_option( 'wpforo_authorization', $this->_authorization );
		$this->email                   = wpforo_get_option( 'wpforo_email', $this->_email );
		$_antispam                     = $this->_antispam;
		$_antispam['limited_file_ext'] = $_antispam['exclude_file_ext'] = [];
		$this->antispam                = wpforo_get_option( 'wpforo_antispam', $_antispam );
		$this->akismet                 = wpforo_get_option( 'wpforo_akismet', $this->_akismet );
		$this->recaptcha               = wpforo_get_option( 'wpforo_recaptcha', $this->_recaptcha );
		$this->buddypress              = wpforo_get_option( 'wpforo_buddypress', $this->_buddypress );
		$this->um                      = wpforo_get_option( 'wpforo_um', $this->_um );
		$this->legal                   = wpforo_get_option( 'wpforo_legal', $this->_legal );
		
		if( ( $this->profiles['profile'] === 'bp' && ! class_exists( 'BP_Component' ) ) || ( $this->profiles['profile'] === 'um' && ! function_exists( 'UM' ) ) ) {
			$this->profiles['profile']       = 'default';
			$this->profiles['default_cover'] = ( trim( (string) $this->profiles['default_cover'] ) ?: $this->_profiles['default_cover'] );
			wpforo_update_option( 'wpforo_profiles', $this->profiles );
		}
		
		do_action( 'wpforo_settings_after_init_base', $this );
	}
	
	private function init() {
		$this->board         = wpforo_get_option( 'board', $this->_board );
		$this->forums        = wpforo_get_option( 'forums', $this->_forums );
		$this->topics        = wpforo_get_option( 'topics', $this->_topics );
		$this->posting       = wpforo_get_option( 'posting', $this->_posting );
		$this->components    = wpforo_get_option( 'components', $this->_components );
		$this->tags          = wpforo_get_option( 'tags', $this->_tags );
		$this->subscriptions = wpforo_get_option( 'subscriptions', $this->_subscriptions );
		$this->notifications = wpforo_get_option( 'notifications', $this->_notifications );
		$this->logging       = wpforo_get_option( 'logging', $this->_logging );
		$this->seo           = wpforo_get_option( 'seo', $this->_seo );
		$this->rss           = wpforo_get_option( 'rss', $this->_rss );
		$this->social        = wpforo_get_option( 'social', $this->_social );
		$this->styles        = wpforo_get_option( 'styles_' . WPF()->tpl->theme, $this->_styles );
		
		if( ! $this->seo['seo_profile'] ) $this->seo['members_sitemap'] = false;
		
		do_action( 'wpforo_settings_after_init', $this );
	}
	
	public function export( $groups = [] ) {
		if( empty( $groups ) ) {
			if( is_null( $this->info ) ) $this->init_info();
			$groups = array_merge( array_keys( $this->info->core ), array_keys( $this->info->addons ) );
		}
		
		if( ! empty( $groups ) ) {
			$settings = [];
			foreach( $groups as $group ) {
				if( ! empty( $this->$group ) ) {
					$settings[ $group ] = $this->$group;
				}
			}
			
			$json = json_encode( $settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES );
			
			header( 'Content-Type: application/json' );
			header( 'Content-Disposition: attachment; filename="wpforo-settings-' . date( 'Y-m-d_H-i-s' ) . '.json"' );
			header( 'Content-Length: ' . strlen( $json ) );
			echo $json;
			exit();
		}
	}
	
	public function import( $json, $groups = null ) {
		if( $json ) {
			$settings = json_decode( $json, true );
			
			if( is_array( $settings ) && ! empty( $settings ) ) {
				if( is_null( $this->info ) ) $this->init_info();
				
				foreach( $settings as $group => $setting ) {
					if( ! is_null( $groups ) && ! in_array( $group, $groups, true ) ) continue;
					
					if( wpfkey( $this->info->core, $group, 'base' ) ) {
						$is_base = $this->info->core[ $group ]['base'];
					} elseif( wpfkey( $this->info->addons, $group, 'base' ) ) {
						$is_base = $this->info->addons[ $group ]['base'];
					}
					
					if( isset( $is_base ) ) {
						$option_name = preg_replace( '#^wpforo_#iu', '', $group, 1 );
						if( $option_name === 'profiles' && wpfval( $setting, 'default_cover' ) ) {
							$setting['default_cover'] = preg_replace( '#^.*?/wp-content/uploads/wpforo/covers/#iu', WPF()->folders['covers']['url//'] . '/', $setting['default_cover'], 1 );
						}
						if( $is_base ) $option_name = 'wpforo_' . $option_name;
						wpforo_update_option( $option_name, $setting, false );
					}
					
				}
			}
			
		}
	}
}
